package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.parsing;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.TemporalAccessor;
import java.util.List;
import java.util.Locale;
import java.util.function.Supplier;

import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CombinedDateParser {
    private static final Logger LOG = LoggerFactory.getLogger(CombinedDateParser.class);

    private final ImmutableList<Supplier<DateTimeFormatter>> formatters;
    private final Locale locale;

    public CombinedDateParser(Locale locale, List<Supplier<DateTimeFormatterBuilder>> builders) {
        this.locale = locale;

        formatters = builders.stream()
                .map((dtfBuilder) -> (Supplier<DateTimeFormatter>) () -> buildWithLocale(dtfBuilder.get()))
                .collect(ImmutableList.toImmutableList());
    }

    public Instant parseStringDate(String stringDate) {
        return LocalDateTime.from(tryParseWithAnyFormatter(stringDate))
                .toInstant(OffsetDateTime.now().getOffset());
    }

    private TemporalAccessor tryParseWithAnyFormatter(String stringDate) {
        for (Supplier<DateTimeFormatter> f : formatters) {
            DateTimeFormatter formatter = f.get();

            try {
                return formatter.parse(stringDate);
            } catch (DateTimeParseException e) {
                LOG.trace("Attempt parsing " + stringDate + " with " + formatter + " failed");
            }
        }

        LOG.error("Unable to parse date with any parser " + stringDate);
        throw new DateTimeParseException("Unable to parse date with any parser " + stringDate, stringDate, 0);
    }

    private DateTimeFormatter buildWithLocale(DateTimeFormatterBuilder builder) {
        return builder.toFormatter(locale);
    }
}
