package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium;

import java.util.concurrent.atomic.AtomicReference;

import org.openqa.selenium.By;
import org.openqa.selenium.JavascriptExecutor;
import org.openqa.selenium.NoSuchElementException;
import org.openqa.selenium.OutputType;
import org.openqa.selenium.StaleElementReferenceException;
import org.openqa.selenium.TimeoutException;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.openqa.selenium.support.ui.ExpectedConditions;
import org.openqa.selenium.support.ui.WebDriverWait;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.mail.WebElementState;

public class SeleniumDriver {
    // https://stat.yandex-team.ru/Dashboard/MailDashboard?tab=1006f
    // 95 percentile
    public static final int FAIL_SAFE_DELAY_SECONDS = 30;
    private static final Logger LOG = LoggerFactory.getLogger(SeleniumDriver.class);
    private final WebDriverFactory driverFactory;
    private final AtomicReference<RemoteWebDriver> driver = new AtomicReference<RemoteWebDriver>();

    public SeleniumDriver(WebDriverFactory driverFactory) {
        this.driverFactory = driverFactory;
    }

    public synchronized void connect() {
        driver.set(driverFactory.createWebDriver());
    }

    public synchronized WebElementState getElementState(By by) {
        return new WebElementState(by, getOrNull(by));
    }

    public synchronized void waitForElementUpdate(WebElementState state, int timeoutSeconds) {
        LOG.trace("Waiting for element to update: " + state.toString());

        WebDriverWait updateWait = new WebDriverWait(driver.get(), timeoutSeconds);
        try {
            updateWait.until(webDriver -> {
                if (state.getStatedElement() != null) {
                    return ExpectedConditions.stalenessOf(state.getStatedElement()).apply(webDriver);
                } else {
                    return getOrNull(state.getLocator()) != null;
                }
            });
        } catch (TimeoutException e) {
            LOG.trace("Timeout waiting for update " + state);
        }
    }

    public synchronized String getCurrentUrl() {
        return driver.get().getCurrentUrl();
    }

    private WebElement getOrNull(By by) {
        try {
            return driver.get().findElement(by);
        } catch (NoSuchElementException e) {
            return null;
        }
    }

    public synchronized void disconnect() {
        LOG.debug("Quitting driver " + driver.toString());
        try {
            driver.get().quit();
        } catch (Exception e) {
            LOG.debug("Exception happened on quiting driver " + e.getClass().getSimpleName() + " " + e.getMessage());
        }
        driver.set(null);
    }

    public boolean isConnected() {
        return driver.get() != null;
    }

    public synchronized void getUrl(String url) {
        LOG.trace("Getting url on driver " + url);
        driver.get().get(url);
    }

    public synchronized void waitForJS() {
        LOG.trace("Waiting for JavaScript completion on " + driver.get().getCurrentUrl());
        WebDriverWait javaScriptWait = new WebDriverWait(driver.get(), FAIL_SAFE_DELAY_SECONDS);
        javaScriptWait.until(
                webDriver -> ((JavascriptExecutor) webDriver)
                        .executeScript("return document.readyState")
                        .equals("complete")
        );
    }

    public WebPageRepresentation getWebPage() {
        String html =
                driver.get().executeScript("return document.getElementsByTagName('html')[0].innerHTML").toString();
        byte[] screenshot = driver.get().getScreenshotAs(OutputType.BYTES);
        return new WebPageRepresentation(html, screenshot);
    }

    public boolean hasElement(By by) {
        try {
            return driver.get().findElement(by) != null;
        } catch (NoSuchElementException e) {
            return false;
        }
    }

    public synchronized void click(By by) {
        LOG.trace("Clicking on " + by.toString());
        WebElement clickable = getDisplayedElementWithWaiting(by);
        clickable.click();
    }

    public synchronized void sendText(By by, String text) {
        LOG.trace("Writing " + text + " to " + by.toString());
        WebElement input = getDisplayedElementWithWaiting(by);
        if (!input.getAttribute("value").isEmpty()) {
            input.clear();
        }

        input.sendKeys(text);
    }

    public synchronized void sendTextWithoutClearing(By by, String text) {
        LOG.trace("Writing " + text + " to " + by.toString() + " without cleaning");
        WebElement input = getDisplayedElementWithWaiting(by);
        input.sendKeys(text);
    }

    public synchronized void clear(By by) {
        LOG.trace("Clearing  " + by.toString());
        WebElement input = getDisplayedElementWithWaiting(by);
        input.clear();
    }

    private WebElement getDisplayedElementWithWaiting(By by) {
        LOG.trace("Waiting for the element to become visible " + by.toString());
        waitForJS();
        WebDriverWait wait = new WebDriverWait(driver.get(), FAIL_SAFE_DELAY_SECONDS);

        wait.until(webDriver -> {
                try {
                    WebElement element = webDriver.findElement(by);
                    return element != null && element.isDisplayed();
                } catch (StaleElementReferenceException e) {
                    LOG.info("Stale element reference exception, keep trying for selector: " + by);
                    return false;
                }
            });

        return driver.get().findElement(by);
    }
}
