package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.mail;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;

import org.openqa.selenium.By;

import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.CrawlerInternalException;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.SeleniumDriver;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.WebCredentials;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.WebPageRepresentation;

public class SeleniumMailClient {
    private final SeleniumDriver driver;

    private final WebClientControlsXPaths xPaths;

    private final LoginSequence loginSequence;
    private final WebCredentials credentials;

    private final AtomicBoolean loggedIn = new AtomicBoolean(false);

    private final String defaultUrl;

    public SeleniumMailClient(
            SeleniumDriver driver,
            WebClientControlsXPaths xPaths,
            LoginSequence loginSequence,
            WebCredentials credentials, String defaultUrl)
    {
        this.driver = driver;
        this.xPaths = xPaths;
        this.loginSequence = loginSequence;
        this.credentials = credentials;
        this.defaultUrl = defaultUrl;
    }

    public void login() {
        if (!driver.isConnected()) {
            driver.connect();
        }
        loginSequence.login(defaultUrl, driver, credentials);
        loggedIn.set(true);
    }

    public void logout() {
        if (driver.isConnected()) {
            driver.disconnect();
        }
        loggedIn.set(false);
    }

    public boolean isLoggedIn() {
        return loggedIn.get();
    }

    public void typeInSearchField(String text) {
        checkDriverConnection();
        doOnExistentWithDefault(xPaths.getSearchFieldXPath(), by -> driver.sendTextWithoutClearing(by, text));
    }

    public void clearSearchField() {
        checkDriverConnection();
        doOnExistentWithDefault(xPaths.getSearchFieldXPath(), driver::clear);
    }

    public void clickOnSearchButton() {
        checkDriverConnection();
        doOnExistentWithDefault(xPaths.getSearchButtonXPath(), driver::click);
    }

    public WebElementState getSearchResultState() {
        return driver.getElementState(By.xpath(xPaths.getSearchResultXPath()));
    }

    public WebElementState getSuggestResultState() {
        return driver.getElementState(By.xpath(xPaths.getSuggestResultXPath()));
    }

    public void waitForStateUpdate(WebElementState state, int timeoutSeconds) {
        driver.waitForElementUpdate(state, timeoutSeconds);
    }

    public void waitForPageLoading() {
        driver.waitForJS();
    }

    public WebPageRepresentation getWebPage() {
        checkDriverConnection();
        return driver.getWebPage();
    }

    private void doOnExistentWithDefault(String xPath, Consumer<By> action) {
        if (driver.hasElement(By.xpath(xPath))) {
            action.accept(By.xpath(xPath));
        } else {
            driver.getUrl(defaultUrl);
            action.accept(By.xpath(xPath));
        }
    }

    private void checkDriverConnection() {
        if (!driver.isConnected()) {
            throw new CrawlerInternalException("Driver is not connected");
        }
    }
}
