package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.mail;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Path;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.personal.mail.search.metrics.scraper.services.account.AccountProperties;
import ru.yandex.personal.mail.search.metrics.scraper.services.account.AccountPropertiesLoader;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.SeleniumDriverFactory;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.WebCredentials;

public class SeleniumMailClientFactory {
    private static final Logger LOG = LoggerFactory.getLogger(SeleniumMailClientFactory.class);

    private final String controlsXPathsResourceLocation;

    private final SeleniumDriverFactory driverFactory;
    private final LoginSequence loginSequence;
    private final ObjectMapper mapper;

    public SeleniumMailClientFactory(
            String controlsXPathsResourceLocation,
            SeleniumDriverFactory driverFactory,
            LoginSequence loginSequence,
            ObjectMapper mapper)
    {
        this.controlsXPathsResourceLocation = controlsXPathsResourceLocation;
        this.driverFactory = driverFactory;
        this.loginSequence = loginSequence;
        this.mapper = mapper;
    }

    public SeleniumMailClient createMailClient(Path accPath, String credential) {
        AccountProperties properties = AccountPropertiesLoader.loadProperties(accPath);
        WebCredentials credentials = readCredentials(accPath.resolve(credential));
        WebClientControlsXPaths xPaths = loadControlsXPathsFromResources();

        return new SeleniumMailClient(
                driverFactory.createDriver(),
                xPaths,
                loginSequence,
                credentials,
                properties.getHostUrl()
        );
    }

    private WebCredentials readCredentials(Path path) {
        try {
            return mapper.readValue(path.toFile(), WebCredentials.class);
        } catch (IOException e) {
            LOG.error("Can not read Selenium credentials: " + path);
            throw new UncheckedIOException(e);
        }
    }

    private WebClientControlsXPaths loadControlsXPathsFromResources() {
        InputStream is = getClass().getClassLoader().getResourceAsStream(controlsXPathsResourceLocation);
        try {
            return mapper.readValue(is, WebClientControlsXPaths.class);
        } catch (IOException e) {
            LOG.warn("Can not load controls xpaths from resources", e);
            throw new UncheckedIOException(e);
        }
    }
}
