package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.mail;

import com.google.common.base.Stopwatch;
import org.openqa.selenium.WebDriverException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.personal.mail.search.metrics.scraper.model.crawling.CrawlingMeta;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.ArchivedMailScrapingResult;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.search.MailSearchResult;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.search.MailSearchScrapedData;
import ru.yandex.personal.mail.search.metrics.scraper.model.query.SearchQuery;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.ArchiveEntry;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.response.html.HtmlResponseRepository;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.screenshot.ScreenshotRepository;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.MailSearchSystem;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.CrawlerInternalException;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.Crawling;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.parsing.MailSearchParser;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.CrawlingResult;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.WebPageRepresentation;

public class SeleniumMailSearchSystem implements MailSearchSystem {
    private static final Logger LOG = LoggerFactory.getLogger(SeleniumMailSearchSystem.class);

    private static final int RETRY_TIMES = 2;
    private static final int SEARCH_RESULT_UPDATE_TIMEOUT = 4;

    private final MailSearchParser parser;
    private final SeleniumMailClient mailClient;

    private final HtmlResponseRepository responseRepository;
    private final ScreenshotRepository screenshotRepository;

    private final String name;

    public SeleniumMailSearchSystem(MailSearchParser parser,
            SeleniumMailClient mailClient,
            HtmlResponseRepository responseRepository,
            ScreenshotRepository screenshotRepository,
            String name)
    {
        this.parser = parser;
        this.mailClient = mailClient;
        this.responseRepository = responseRepository;
        this.screenshotRepository = screenshotRepository;
        this.name = name;
    }

    @Override
    public MailSearchResult search(SearchQuery query) {
        for (int i = 0; i < RETRY_TIMES; i++) {
            try {
                return searchWithoutRetries(query);
            } catch (CrawlerInternalException e) {
                LOG.warn("Exception on " + i + " attempt", e);
            }
        }
        return searchWithoutRetries(query);
    }

    private MailSearchResult searchWithoutRetries(SearchQuery query) {
        CrawlingResult crawlingResult;
        MailSearchScrapedData searchScrapedData;

        try {
            crawlingResult = getPageTheadSafe(query.getText());
            searchScrapedData = parser.parse(crawlingResult.getWebPageRepresentation().getHtml());
        } catch (WebDriverException e) {
            LOG.warn("Webdriver exception, disconnecting " + e.getMessage());
            mailClient.logout();
            throw new CrawlerInternalException(e);
        }

        ArchivedMailScrapingResult
                archivedMailScrapingResult = archiveResult(crawlingResult.getWebPageRepresentation(), query);
        return MailSearchResult
                .successful(query, searchScrapedData, archivedMailScrapingResult, crawlingResult.getCrawlingMeta());
    }

    private ArchivedMailScrapingResult archiveResult(WebPageRepresentation page, SearchQuery query) {
        ArchiveEntry response = responseRepository.save(page.getHtml(),
                name + "_" + query.getText() + "_.html");
        ArchiveEntry screenshot = screenshotRepository.save(page.getScreenshot(),
                name + "_" + query.getText() + "_.png");

        return new ArchivedMailScrapingResult(screenshot, response);
    }

    private synchronized CrawlingResult getPageTheadSafe(String query) {
        if (!mailClient.isLoggedIn()) {
            mailClient.login();
        }
        mailClient.clearSearchField();

        Stopwatch stopwatch = Stopwatch.createStarted();
        WebElementState searchState = mailClient.getSearchResultState();
        mailClient.typeInSearchField(query);
        mailClient.clickOnSearchButton();
        mailClient.waitForPageLoading();
        mailClient.waitForStateUpdate(searchState, SEARCH_RESULT_UPDATE_TIMEOUT);
        WebPageRepresentation representation = mailClient.getWebPage();
        stopwatch.stop();

        CrawlingMeta crawlingMeta = Crawling.meta(stopwatch, representation);
        return new CrawlingResult(representation, crawlingMeta);
    }

    @Override
    public void logout() {
        mailClient.logout();
    }
}
