package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.mail;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import com.google.common.base.Stopwatch;
import org.openqa.selenium.WebDriverException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.personal.mail.search.metrics.scraper.model.mail.ArchivedMailScrapingResult;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.suggest.MailSuggestPartResult;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.suggest.MailSuggestResult;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.suggest.MailSuggestSnippet;
import ru.yandex.personal.mail.search.metrics.scraper.model.query.SearchQuery;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.ArchiveEntry;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.response.html.HtmlResponseRepository;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.screenshot.ScreenshotRepository;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.MailSuggestSystem;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.CrawlerInternalException;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.Crawling;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.suggest.querysplitters.QueryPart;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.suggest.querysplitters.QuerySplitter;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.parsing.MailSuggestParser;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.CrawlingResult;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.selenium.WebPageRepresentation;

public class SeleniumMailSuggestSystem implements MailSuggestSystem {
    private static final Logger LOG = LoggerFactory.getLogger(SeleniumMailSuggestSystem.class);

    private static final int RETRY_TIMES = 2;

    private final int suggestTimout;
    private final String name;

    private final MailSuggestParser parser;
    private final SeleniumMailClient mailClient;

    private final HtmlResponseRepository responseRepository;
    private final ScreenshotRepository screenshotRepository;

    private final QuerySplitter querySplitter;

    public SeleniumMailSuggestSystem(
            int suggestTimout,
            String name,
            MailSuggestParser parser,
            SeleniumMailClient mailClient,
            HtmlResponseRepository responseRepository,
            ScreenshotRepository screenshotRepository,
            QuerySplitter querySplitter)
    {
        this.suggestTimout = suggestTimout;
        this.name = name;
        this.parser = parser;
        this.mailClient = mailClient;
        this.responseRepository = responseRepository;
        this.screenshotRepository = screenshotRepository;
        this.querySplitter = querySplitter;
    }


    @Override
    public MailSuggestResult suggest(SearchQuery query) {
        for (int i = 0; i < RETRY_TIMES; i++) {
            try {
                return searchForAll(query);
            } catch (WebDriverException e) {
                LOG.warn("Webdriver exception on " + i + " attempt, reconnecting " + e.getMessage());
                mailClient.logout();
            } catch (CrawlerInternalException e) {
                LOG.warn("Exception on " + i + " attempt ", e);
            }
        }
        return searchForAll(query);
    }

    private synchronized MailSuggestResult searchForAll(SearchQuery query) {
        prepareForSuggest();

        List<MailSuggestPartResult> suggestResults =
                StreamSupport.stream(querySplitter.splitQuery(query).spliterator(), false)
                        .map(queryPart -> getSingleSuggestResult(query, queryPart))
                        .collect(Collectors.toList());

        return MailSuggestResult.successful(query, suggestResults);
    }

    @Override
    public void logout() {
        LOG.trace("Logging out of selenium mail client");
        mailClient.logout();
    }

    private MailSuggestPartResult getSingleSuggestResult(SearchQuery query, QueryPart queryPart) {
        CrawlingResult crawlingResult = getResultWithNewSymbols(queryPart.getDiff());

        List<MailSuggestSnippet> snippets = parser.parse(crawlingResult.getWebPageRepresentation().getHtml());
        ArchivedMailScrapingResult
                archivedMailScrapingResult = archiveResult(crawlingResult.getWebPageRepresentation(), query, queryPart);

        return MailSuggestPartResult.successful(query, queryPart,
                crawlingResult.getCrawlingMeta(), archivedMailScrapingResult, snippets);
    }

    private ArchivedMailScrapingResult archiveResult(WebPageRepresentation page, SearchQuery searchQuery,
            QueryPart queryPart)
    {
        ArchiveEntry response = responseRepository.save(page.getHtml(),
                name + "_" + searchQuery.getText() + queryPart.getPart() + "_.html");
        ArchiveEntry screenshot = screenshotRepository.save(page.getScreenshot(),
                name + "_" + searchQuery.getText() + queryPart.getPart() + "_.png");

        return new ArchivedMailScrapingResult(screenshot, response);
    }

    private void prepareForSuggest() {
        if (!mailClient.isLoggedIn()) {
            mailClient.login();
        }
        mailClient.clearSearchField();
    }

    private CrawlingResult getResultWithNewSymbols(String symbols) {
        Stopwatch stopwatch = Stopwatch.createStarted();
        WebElementState suggestState = mailClient.getSuggestResultState();
        mailClient.typeInSearchField(symbols);
        mailClient.waitForStateUpdate(suggestState, suggestTimout);
        WebPageRepresentation representation = mailClient.getWebPage();
        stopwatch.stop();

        return new CrawlingResult(representation, Crawling.meta(stopwatch, representation));
    }
}
