package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.gapi;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.api.services.gmail.Gmail;
import com.google.api.services.gmail.model.ListMessagesResponse;
import com.google.api.services.gmail.model.Message;
import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.retry.annotation.Retryable;

import ru.yandex.personal.mail.search.metrics.scraper.model.crawling.CrawlingMeta;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.ArchivedMailScrapingResult;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.search.MailSearchMessageSnippet;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.search.MailSearchResult;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.search.MailSearchScrapedData;
import ru.yandex.personal.mail.search.metrics.scraper.model.query.SearchQuery;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.ArchiveEntry;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.response.json.JsonResponseRepository;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.MailSearchSystem;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.CrawlerException;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.CrawlerInternalException;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.Crawling;


public class GApiSearchSystem implements MailSearchSystem {
    private static final Logger LOG = LoggerFactory.getLogger(GApiSearchSystem.class);
    private static final String MAIL_USER = "me";

    private final ExecutorService worker = Executors.newCachedThreadPool();
    private final Gmail gmail;
    private final GApiMessageToMessageSearchSnippetConverter converter;

    private final JsonResponseRepository responseRepository;
    private final ObjectMapper objectMapper;

    public GApiSearchSystem(Gmail gmail, GApiMessageToMessageSearchSnippetConverter converter,
            JsonResponseRepository responseRepository, ObjectMapper objectMapper)
    {
        this.gmail = gmail;
        this.converter = converter;
        this.responseRepository = responseRepository;
        this.objectMapper = objectMapper;
    }

    @Override
    public MailSearchResult search(SearchQuery query) {
        LOG.trace("Searching for " + query.getText());

        Stopwatch stopwatch = Stopwatch.createStarted();
        ListMessagesResponse response = searchMessageIds(query.getText());
        loadFullMessages(response);
        stopwatch.stop();

        List<MailSearchMessageSnippet> snippets = response.getMessages().stream()
                .map(converter::convert)
                .collect(Collectors.toList());

        String responseJson = writeResponseToString(response);
        ArchivedMailScrapingResult archivedMailScrapingResult = archiveResult(query, responseJson);
        CrawlingMeta crawlingMeta = Crawling.meta(stopwatch, responseJson);


        return MailSearchResult.successful(query,
                new MailSearchScrapedData(Math.toIntExact(response.getResultSizeEstimate()), snippets),
                archivedMailScrapingResult,
                crawlingMeta);
    }

    private String writeResponseToString(ListMessagesResponse response) {
        try {
            return objectMapper.writeValueAsString(response);
        } catch (JsonProcessingException e) {
            throw new CrawlerInternalException(e);
        }
    }

    private ArchivedMailScrapingResult archiveResult(SearchQuery query, String json) {
        ArchiveEntry response = responseRepository.save(json, "gapi_" + query.getText() + "_.json");
        return ArchivedMailScrapingResult.responseOnly(response);
    }

    private void loadFullMessages(ListMessagesResponse response) {
        if (response.getMessages() == null) {
            response.setMessages(Collections.emptyList());
            return;
        }

        List<Message> messages = response.getMessages().stream()
                .map((msg) -> worker.submit(() -> getMsg(msg.getId())))
                .map(this::msgFromFuture)
                .collect(Collectors.toList());

        response.setMessages(messages);
    }

    @Retryable(value = {Exception.class}, maxAttempts = 5)
    private ListMessagesResponse searchMessageIds(String query) {
        try {
            return gmail.users().messages().list(MAIL_USER).setQ(query).execute();
        } catch (IOException e) {
            String message = "Can not access google api to search with query " + query + " error: " + e.getMessage();
            LOG.warn(message);
            throw new CrawlerException(message, e);
        }
    }

    private Message msgFromFuture(Future<Message> future) {
        try {
            return future.get();
        } catch (InterruptedException | ExecutionException e) {
            throw new CrawlerException(e);
        }
    }

    @Retryable(value = {Exception.class}, maxAttempts = 5)
    private Message getMsg(String id) {
        try {
            return gmail.users().messages().get(MAIL_USER, id).execute();
        } catch (IOException e) {
            String message = "Can not access google api to resolve message by id " + id + " error: " + e.getMessage();
            LOG.warn(message);
            throw new CrawlerException(message, e);
        }
    }
}
