package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.mweb.search;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.util.Locale;
import java.util.function.Supplier;

import com.google.common.collect.Lists;

import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.parsing.CombinedDateParser;

/**
 * Date parser for mail.ru mail web interface.
 * <p>
 * There are 2 major date representations used in a mail.ru service:
 * date with omitted (current) year and full date with year.
 * Also there are 2 extra formats for recent messages received today and yesterday,
 * but those a rarely used because mailbox is artificially generated.
 */
public class MailruDateParser {
    private static final Supplier<DateTimeFormatterBuilder> todayParser = () -> new DateTimeFormatterBuilder()
            .appendPattern("Сегодня, H:mm")
            .parseDefaulting(ChronoField.DAY_OF_MONTH, LocalDateTime.now().getDayOfMonth())
            .parseDefaulting(ChronoField.MONTH_OF_YEAR, LocalDateTime.now().getMonthValue())
            .parseDefaulting(ChronoField.YEAR, LocalDateTime.now().getYear());

    private static final Supplier<DateTimeFormatterBuilder> yesterdayParser = () -> new DateTimeFormatterBuilder()
            .appendPattern("Вчера, H:mm")
            .parseDefaulting(ChronoField.DAY_OF_MONTH, LocalDateTime.now().minusDays(1).getDayOfMonth())
            .parseDefaulting(ChronoField.MONTH_OF_YEAR, LocalDateTime.now().minusDays(1).getMonthValue())
            .parseDefaulting(ChronoField.YEAR, LocalDateTime.now().minusDays(1).getYear());

    private static final Supplier<DateTimeFormatterBuilder> partialDateDtf = () -> new DateTimeFormatterBuilder()
            .appendPattern("d MMMM, H:mm")
            .parseDefaulting(ChronoField.YEAR, LocalDateTime.now().getYear());

    private static final Supplier<DateTimeFormatterBuilder> fullDateDtf = () -> new DateTimeFormatterBuilder()
            .appendPattern("d MMMM yyyy, H:mm");

    private static final CombinedDateParser dateParser = new CombinedDateParser(new Locale("RU", "ru"),
            Lists.newArrayList(partialDateDtf, fullDateDtf, yesterdayParser, todayParser));

    public Instant parseStringDate(String stringDate) {
        return dateParser.parseStringDate(stringDate);
    }
}
