package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.mweb.search;

import java.time.Instant;
import java.util.List;
import java.util.stream.Collectors;

import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.personal.mail.search.metrics.scraper.model.mail.search.MailSearchMessageSnippet;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.search.MailSearchScrapedData;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.parsing.MailSearchParser;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.parsing.WebParseTools;

@Service
public class MailruSearchParser implements MailSearchParser {
    private static final Logger LOG = LoggerFactory.getLogger(MailruSearchParser.class);

    private static final String SEARCH_RESULT_BLOCK_SELECTOR = "div[class=b-datalist__body]";
    private static final String SEARCH_RESULT_SELECTOR = "div[data-bem=b-datalist__item]";
    private static final String MESSAGE_DATE_SELECTOR =
            "span[class^=b-datalist__item__value b-datalist__item__date__value]";
    private static final String MESSAGE_SENDER_SELECTOR = "div[class*=b-datalist__item__addr]";
    private static final String MESSAGE_SUBJECT_SELECTOR = "div[class*=b-datalist__item__subj]";
    private static final String MESSAGE_SNIPPET_SELECTOR = "span[class*=b-datalist__item__subj__snippet]";
    private static final String SERP_SIZE_SELECTOR = "span[class=b-nav__item__count]";

    private MailruDateParser dateParser = new MailruDateParser();

    @Override
    public MailSearchScrapedData parse(String html) {
        LOG.trace("Parsing response");
        Document page = WebParseTools.prepareVisiblePage(html);

        Element tableBody = page.selectFirst(SEARCH_RESULT_BLOCK_SELECTOR);
        if (tableBody == null) {
            return MailSearchScrapedData.empty();
        }

        int docsFound = parseDocsFound(page);
        List<MailSearchMessageSnippet> snippets = parseSnippets(tableBody);

        return new MailSearchScrapedData(docsFound, snippets);
    }

    private int parseDocsFound(Document page) {
        Elements serpSizeElements = page.select(SERP_SIZE_SELECTOR);
        int sumOfBothSerpSizes = serpSizeElements.stream()
                .mapToInt(e -> Integer.parseInt(e.text()))
                .sum();
        return sumOfBothSerpSizes / 2;
    }

    private List<MailSearchMessageSnippet> parseSnippets(Element table) {
        Elements searchResults = table.select(SEARCH_RESULT_SELECTOR);
        return searchResults.stream()
                .map(this::parseSnippet)
                .collect(Collectors.toList());
    }

    private MailSearchMessageSnippet parseSnippet(Element e) {
        String sender = e.selectFirst(MESSAGE_SENDER_SELECTOR).text();
        String subject = e.selectFirst(MESSAGE_SUBJECT_SELECTOR).ownText();
        String snippet = e.selectFirst(MESSAGE_SNIPPET_SELECTOR).text();
        String stringDate = e.selectFirst(MESSAGE_DATE_SELECTOR).attr("title");
        Instant date = dateParser.parseStringDate(stringDate);

        return new MailSearchMessageSnippet(subject, snippet, sender, date);
    }
}
