package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.yapi.suggest;

import java.io.IOException;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Stopwatch;

import ru.yandex.personal.mail.search.metrics.scraper.model.crawling.CrawlingMeta;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.ArchivedMailScrapingResult;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.suggest.MailSuggestPartResult;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.suggest.MailSuggestResult;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.suggest.MailSuggestSnippet;
import ru.yandex.personal.mail.search.metrics.scraper.model.query.SearchQuery;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.ArchiveEntry;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.response.json.JsonResponseRepository;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.MailSuggestSystem;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.CrawlerInternalException;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.Crawling;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.suggest.querysplitters.QueryPart;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.suggest.querysplitters.QuerySplitter;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.yapi.YApiUser;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.yapi.response.YApiSuggestItem;

public class YApiSuggestSystem implements MailSuggestSystem {
    private final YApiUser user;
    private final JsonResponseRepository responseRepository;
    private final QuerySplitter querySplitter;
    private final ObjectMapper mapper;

    public YApiSuggestSystem(YApiUser user,
            JsonResponseRepository responseRepository,
            QuerySplitter querySplitter, ObjectMapper mapper)
    {
        this.user = user;
        this.responseRepository = responseRepository;
        this.querySplitter = querySplitter;
        this.mapper = mapper;
    }

    @Override
    public MailSuggestResult suggest(SearchQuery query) {
        List<MailSuggestPartResult> suggestResults = searchEach(query);
        return MailSuggestResult.successful(query, suggestResults);
    }

    private List<MailSuggestPartResult> searchEach(SearchQuery query) {
        return StreamSupport.stream(querySplitter.splitQuery(query).spliterator(), false)
                .map(qp -> searchSplit(query, qp))
                .collect(Collectors.toList());
    }

    private MailSuggestPartResult searchSplit(SearchQuery fullQuery, QueryPart queryPart) {
        Stopwatch stopwatch = Stopwatch.createStarted();
        String json = user.suggest(queryPart.getPart());
        stopwatch.stop();

        List<MailSuggestSnippet> snippets = parseSuggest(json);
        ArchivedMailScrapingResult archivedMailScrapingResult = archiveResult(json, fullQuery, queryPart);
        CrawlingMeta statistics = Crawling.meta(stopwatch, json);

        return MailSuggestPartResult.successful(fullQuery, queryPart, statistics, archivedMailScrapingResult, snippets);
    }

    private ArchivedMailScrapingResult archiveResult(String json, SearchQuery query, QueryPart queryPart) {
        ArchiveEntry response = responseRepository.save(json,
                "yapi-suggest_" + query.getText() + "_" + queryPart.getPart() + "_.json");
        return ArchivedMailScrapingResult.responseOnly(response);
    }

    private List<MailSuggestSnippet> parseSuggest(String suggestJson) {
        return getSuggestValue(suggestJson).stream()
                .map(msg -> MailSuggestSnippet.simple(msg.getShowText()))
                .collect(Collectors.toList());
    }

    private List<YApiSuggestItem> getSuggestValue(String suggestJson) {
        try {
            return mapper.readValue(suggestJson, new TypeReference<List<YApiSuggestItem>>() {
            });
        } catch (IOException e) {
            throw new CrawlerInternalException(e);
        }
    }
}
