package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.ymob;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Path;

import com.fasterxml.jackson.databind.ObjectMapper;
import okhttp3.OkHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;

import ru.yandex.personal.mail.search.metrics.scraper.services.account.AccountProperties;
import ru.yandex.personal.mail.search.metrics.scraper.services.account.AccountPropertiesLoader;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.ymob.api.UserApi;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.ymob.api.UserApiV2;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.ymob.api.request.YMobClient;

@Service
public class YMobUserLoader {
    private static final Logger LOG = LoggerFactory.getLogger(YMobUserLoader.class);

    private static final String CREDENTIAL = "credentials";
    private static final YMobClient CLIENT = YMobClient.DEFAULT;
    private final ObjectMapper mapper;

    public YMobUserLoader(ObjectMapper mapper) {
        this.mapper = mapper;
    }

    public static boolean validateCredentialName(String credentialName) {
        return CREDENTIAL.equals(credentialName);
    }

    public YMobUser loadUser(Path accPath) {
        AccountProperties properties = AccountPropertiesLoader.loadProperties(accPath);
        YMobCredentials credentials = readCredentials(accPath);

        Retrofit retrofit = yMobRetrofit(properties.getHostUrl());
        UserApi api = retrofit.create(UserApi.class);
        UserApiV2 apiV2 = retrofit.create(UserApiV2.class);

        return new YMobUser(credentials, api, apiV2, CLIENT);
    }

    private Retrofit yMobRetrofit(String url) {
        url = ensureTrailingSlash(url);
        OkHttpClient.Builder httpClient = new OkHttpClient.Builder();
        return new Retrofit.Builder()
                .baseUrl(url)
                .addConverterFactory(GsonConverterFactory.create())
                .client(httpClient.build())
                .build();
    }

    private String ensureTrailingSlash(String url) {
        if (!url.endsWith("/")) {
            url += "/";
        }
        return url;
    }

    private YMobCredentials readCredentials(Path accPath) {
        try {
            return mapper.readValue(accPath.resolve(CREDENTIAL).toFile(), YMobCredentials.class);
        } catch (IOException e) {
            LOG.error("Error reading YMobCredentials", e);
            throw new UncheckedIOException(e);
        }
    }
}
