package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.ymob.search;

import java.io.IOException;
import java.time.Instant;
import java.util.List;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Stopwatch;

import ru.yandex.personal.mail.search.metrics.scraper.model.crawling.CrawlingMeta;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.ArchivedMailScrapingResult;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.search.MailSearchMessageSnippet;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.search.MailSearchResult;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.search.MailSearchScrapedData;
import ru.yandex.personal.mail.search.metrics.scraper.model.query.SearchQuery;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.ArchiveEntry;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.response.json.JsonResponseRepository;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.MailSearchSystem;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.CrawlerInternalException;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.Crawling;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.ymob.YMobUser;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.ymob.api.response.SearchResponse;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.ymob.api.response.YaMessage;

public class YMobSearchSystem implements MailSearchSystem {
    private final YMobUser user;
    private final JsonResponseRepository responseRepository;
    private final ObjectMapper mapper;

    public YMobSearchSystem(YMobUser user,
            JsonResponseRepository responseRepository, ObjectMapper mapper)
    {
        this.user = user;
        this.responseRepository = responseRepository;
        this.mapper = mapper;
    }

    @Override
    public MailSearchResult search(SearchQuery query) {
        Stopwatch stopwatch = Stopwatch.createStarted();
        String jsonString = user.search(query.getText());
        List<YaMessage> searchResult = parseResult(jsonString);

        stopwatch.stop();

        List<MailSearchMessageSnippet> snippets = searchResult.stream()
                .map(this::yaMsgToMessageSearchSnippet)
                .collect(Collectors.toList());

        ArchivedMailScrapingResult archivedMailScrapingResult = archiveResult(query, jsonString);
        CrawlingMeta crawlingMeta = Crawling.meta(stopwatch, jsonString);

        return MailSearchResult.successful(
                query,
                new MailSearchScrapedData(searchResult.size(), snippets),
                archivedMailScrapingResult,
                crawlingMeta);
    }

    private ArchivedMailScrapingResult archiveResult(SearchQuery query, String json) {
        ArchiveEntry archiveEntry = responseRepository.save(json, "ymob_" + query.getText() + "_.json");
        return ArchivedMailScrapingResult.responseOnly(archiveEntry);
    }

    private List<YaMessage> parseResult(String json) {
        try {
            return mapper.readValue(json, SearchResponse.class).getMessages();
        } catch (IOException e) {
            throw new CrawlerInternalException(e);
        }
    }

    private MailSearchMessageSnippet yaMsgToMessageSearchSnippet(YaMessage message) {
        String sender = message.getFrom().getName();
        if (sender == null || sender.isEmpty()) {
            sender = message.getFrom().getEmail();
        }

        String subject = message.getSubjText();
        String snippet = message.getFirstLine();
        Instant date = Instant.ofEpochSecond(message.getUtcTimestamp());

        return new MailSearchMessageSnippet(subject, snippet, sender, date);
    }
}
