package ru.yandex.personal.mail.search.metrics.scraper.services.scraping.systems.yweb.search;

import java.time.Instant;
import java.util.List;
import java.util.stream.Collectors;

import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.personal.mail.search.metrics.scraper.model.mail.search.MailSearchMessageSnippet;
import ru.yandex.personal.mail.search.metrics.scraper.model.mail.search.MailSearchScrapedData;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.CrawlerInternalException;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.parsing.MailSearchParser;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.parsing.WebParseTools;

@Service
public class YaSearchParser implements MailSearchParser {
    private static final Logger LOG = LoggerFactory.getLogger(YaSearchParser.class);

    private static final String RESULTS_LIST_SELECTOR = "div[class^=mail-MessageSnippet-Wrapper]";
    private static final String MESSAGE_DATE_SELECTOR = ".mail-MessageSnippet-Item_dateText";

    private static final String MESSAGE_SENDER_SELECTOR = "span[class*=mail-MessageSnippet-FromText]";
    private static final String MESSAGE_SUBJECT_SELECTOR =
            "span[class=mail-MessageSnippet-Item mail-MessageSnippet-Item_subject]";
    private static final String MESSAGE_SNIPPET_SELECTOR = "span[class*=mail-MessageSnippet-Item_firstline]";

    private static final String SERP_SIZE_SELECTOR = "span[class*=mail-MessagesSearchInfo-Title_misc]";
    private static final String EMPTY_SERP_WARNING_SELECTOR = "div[class=b-search-not-found]";
    private static final String SERP_SIZE_1_TEXT = "одно письмо";

    private final YaDateParser dateParser = new YaDateParser();

    @Override
    public MailSearchScrapedData parse(String html) {
        LOG.trace("Parsing response");
        Document page = WebParseTools.prepareVisiblePage(html);

        List<MailSearchMessageSnippet> components = parseSnippets(page);
        int docsFound = parseDocsFound(page);

        return new MailSearchScrapedData(docsFound, components);
    }

    private List<MailSearchMessageSnippet> parseSnippets(Document page) {
        Elements searchResults = page.select(RESULTS_LIST_SELECTOR);
        return searchResults.stream()
                .map(this::parseSnippet)
                .collect(Collectors.toList());
    }

    private MailSearchMessageSnippet parseSnippet(Element element) {
        String dateString = element.selectFirst(MESSAGE_DATE_SELECTOR).attr("title");
        Instant date = dateParser.parseDateString(dateString);
        String sender = element.selectFirst(MESSAGE_SENDER_SELECTOR).text();
        String subject = element.selectFirst(MESSAGE_SUBJECT_SELECTOR).text();
        String snippet = element.selectFirst(MESSAGE_SNIPPET_SELECTOR).text();
        return new MailSearchMessageSnippet(subject, snippet, sender, date);
    }

    private int parseDocsFound(Document page) {
        Element docsCountElement = page.selectFirst(SERP_SIZE_SELECTOR);

        if (docsCountElement == null) {
            assertIsTrulyEmpty(page);
            return 0;
        } else if (manyDocsCount(docsCountElement)) {
            return parseManyDocs(docsCountElement);
        } else if (singleDocFount(docsCountElement)) {
            return 1;
        }

        String message = "Can not parse search size: " + docsCountElement.text();
        LOG.error(message);
        throw new CrawlerInternalException(message);
    }

    private boolean manyDocsCount(Element docsCountElement) {
        return !docsCountElement.text().replaceAll("\\D+", "").isEmpty();
    }

    private int parseManyDocs(Element docsCountElement) {
        return Integer.parseInt(docsCountElement.text().replaceAll("\\D+", ""));
    }

    private boolean singleDocFount(Element docsCountElement) {
        return docsCountElement.text().equals(SERP_SIZE_1_TEXT);
    }

    private void assertIsTrulyEmpty(Document page) {
        Elements emptySerpWarnings = page.select(EMPTY_SERP_WARNING_SELECTOR);
        if (emptySerpWarnings.isEmpty()) {
            String message = "There is no docs count text and search empty warning for ";
            LOG.error(message);
            throw new CrawlerInternalException(message);
        }
    }
}
