import argparse
import json
import logging

from client.api import search_with_retries

logging.basicConfig(format='%(asctime)s %(levelname)s - %(message)s', level=logging.INFO)


def construct_parser():
    parser = argparse.ArgumentParser(description='')
    parser.add_argument('-b', '--basket', required=True)
    parser.add_argument('-s', '--sys', required=True)
    parser.add_argument('-a', '--acc', required=True)
    parser.add_argument('-u', '--url', required=True)
    parser.add_argument('-o', '--out', required=True)
    parser.add_argument('-r', '--retries', required=True)
    parser.add_argument('-t', '--timeout', required=True)
    parser.add_argument('-f', '--accept_failures', required=True)
    return parser


def main():
    args = construct_parser().parse_args()

    logging.info('Starting with arguments %s', args)

    sys = args.sys
    acc = args.acc
    url = args.url
    failure_part = float(args.accept_failures)
    retries = int(args.retries)
    timeout = int(args.timeout)

    with open(args.basket, encoding='utf-8') as b:
        basket = json.load(b)

    errors_amount = 0
    serpset = []
    for query in basket:
        logging.info('Sending query: %s', query['text'])
        try:
            response = search_with_retries(url, sys, acc, query, retries, timeout)
            serpset.append(json.loads(response.text))
            logging.info('ok')
        except Exception as e:
            serpset.append({'type': 'SERP', 'metricTags.failed': True, 'headers': {'errorMessage': str(e)}})
            errors_amount += 1
            if errors_amount / len(basket) > failure_part:
                raise Exception('Too many errors')

    with open(args.out, 'w', encoding='utf-8') as o:
        json.dump(serpset, o, ensure_ascii=False)


if __name__ == '__main__':
    main()
