package ru.yandex.passport;

import java.util.Collections;
import java.util.Set;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.passport.address.AddressId;
import ru.yandex.passport.address.AddressService;
import ru.yandex.passport.address.Locale;
import ru.yandex.passport.address.ServiceConfig;
import ru.yandex.passport.address.config.AddressServicesRights;

public class AddressContext {
    private final ProxySession session;
    private final String userType;
    private final Object userId;
    private final String serviceId;
    private final AddressService service;
    private final JsonType jsonType;
    private final Boolean draft;
    private final Boolean returnUpdated;
    private final boolean taxi;
    private final Locale locale;
    private final boolean convert;
    private final ServiceConfig serviceConfig;

    public AddressContext(final ProxySession session) throws BadRequestException {
        this(session, true);
    }

    public AddressContext(final ProxySession session, final boolean requireUser) throws BadRequestException {
        if (requireUser) {
            this.userType = session.params().getString("user_type", "uid");
            if ("uid".equalsIgnoreCase(userType)) {
                this.userId = session.params().getLong("user_id");
            } else {
                this.userId = session.params().getString("user_id");
            }
        } else {
            this.userType = session.params().getString("user_type", "uid");
            if ("uid".equalsIgnoreCase(userType)) {
                this.userId = session.params().getLong("user_id", null);
            } else {
                this.userId = session.params().getString("user_id", null);
            }
        }

        this.draft = session.params().getBoolean("draft", false);
        this.returnUpdated = session.params().getBoolean("return_updated", true);
        String serviceId = (String) session.context().getAttribute(HttpServer.SESSION_USER);
        if (serviceId == null) {
            this.serviceId = session.params().getString("request_service");
            //throw new BadRequestException("No service id provided");
        } else {
            int index = serviceId.indexOf(':');
            if (index < 0 || index >= serviceId.length() - 1) {
                throw new BadRequestException("Cannot parse service id " + serviceId);
            }
            this.serviceId = serviceId.substring(index + 1);
        }
        this.service = AddressServicesRights.INSTANCE.servicesTvmMap().getOrDefault(this.serviceId, AddressService.UNKNOWN);
        this.serviceConfig = AddressServicesRights.INSTANCE.serviceConfig(service);

        this.taxi = "taxi".equalsIgnoreCase(this.service.serviceName());
        this.session = session;
        session.logger().info("Request service: " + service.serviceName());
        this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        this.locale = parseLocale();
        this.convert = session.params().getBoolean("convert", false);
    }

    public AddressContext(final AddressContext context, final String userType, final Object userId) {
        this.session = context.session;
        this.userType = userType;
        this.userId = userId;
        this.serviceId = context.serviceId;
        this.service = context.service;
        this.jsonType = context.jsonType;
        this.draft = context.draft;
        this.returnUpdated = context.returnUpdated;
        this.taxi = context.taxi;
        this.locale = context.locale;
        this.convert = context.convert;
        this.serviceConfig = context.serviceConfig;
    }

    public Locale locale() {
        return locale;
    }

    public boolean taxi() {
        return taxi;
    }

    public JsonType jsonType() {
        return jsonType;
    }

    public String userType() {
        return userType;
    }

    public Object userId() {
        return userId;
    }

    public Long userIdLong() {
        return (Long) userId;
    }

    public String serviceId() {
        return serviceId;
    }

    public ProxySession session() {
        return session;
    }

    public AddressService service() {
        return service;
    }

    public String serviceName() {
        return service.serviceName();
    }

    public Boolean draft() {
        return draft;
    }

    public Boolean returnUpdated() {
        return returnUpdated;
    }

    public Boolean convert() {
        return convert;
    }

    public Set<AddressService> allowReads() {
        return AddressServicesRights.INSTANCE.allows().get(service).keySet();
    }

    public ServiceConfig serviceConfig() {
        return serviceConfig;
    }

    public boolean modifyAllowed(final AddressService other) {
        return ReadWriteRights.READ_WRITE == AddressServicesRights.INSTANCE.allows()
            .getOrDefault(service, Collections.emptyMap())
            .get(other);
    }

    public ReadWriteRights rights(final AddressId id) {
        return AddressServicesRights.INSTANCE.allows().get(service).get(
            id.ownerService());
    }

    private Locale parseLocale() throws BadRequestException {
        String localeStr = session.params().getOrNull("locale");
        if (localeStr == null) {
            return Locale.RU;
        }
        try {
            return Locale.valueOf(localeStr.toUpperCase(java.util.Locale.ROOT));
        } catch (IllegalArgumentException e) {
            throw new BadRequestException(
                "Failed to parse locale with value " + localeStr);
        }
    }
}
