package ru.yandex.passport;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import io.vertx.core.Vertx;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.client.pg.PgClient;
import ru.yandex.collection.Pattern;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.passport.address.AddressBuilder;
import ru.yandex.passport.address.AddressService;
import ru.yandex.passport.address.AsyncGeocoderClient;
import ru.yandex.passport.address.config.AddressServicesRights;
import ru.yandex.passport.address.config.ImmutableAddressProxyConfig;
import ru.yandex.passport.address.handlers.CreateAddressHandler;
import ru.yandex.passport.address.handlers.DeleteAddressHandler;
import ru.yandex.passport.address.handlers.GetAddressHandler;
import ru.yandex.passport.address.handlers.ListAddressHandler;
import ru.yandex.passport.address.handlers.UpdateAddressHandler;
import ru.yandex.passport.contact.handlers.CreateContactHandler;
import ru.yandex.passport.contact.handlers.DeleteContactHandler;
import ru.yandex.passport.contact.handlers.GetContactHandler;
import ru.yandex.passport.contact.handlers.ListContactHandler;
import ru.yandex.passport.contact.handlers.UpdateContactHandler;
import ru.yandex.passport.storage.HomeWorkDataSyncStorage;
import ru.yandex.passport.storage.MarketPgStorage;
import ru.yandex.passport.storage.PassportDeliveryDataSyncStorage;
import ru.yandex.passport.storage.PassportPgStorage;
import ru.yandex.search.msal.pool.DBConnectionPool;
import ru.yandex.search.msal.pool.ImmutableAddressPgConfig;

public class AddressProxy extends HttpProxy<ImmutableAddressProxyConfig> {
    private final ConcurrentHashMap<String, DBConnectionPool> pools = new ConcurrentHashMap<>();
    private final AsyncGeocoderClient geocoderClient;
    private final AsyncClient homeWorkDatasyncClient;
    private final AsyncClient deliveryDatasyncClient;
    private final BlackboxClient blackboxClient;

    private final Map<AddressService, AddressStorage<AddressBuilder>> addressStorages;
    private final Set<AddressService> passportServices;
    private final Map<AddressService, Set<AddressStorage<AddressBuilder>>> allowedReads;
    private final Map<AddressService, Set<AddressStorage<AddressBuilder>>> allowedWrites;
    private final PgClient pgClient;
    public AddressProxy(final ImmutableAddressProxyConfig config) throws IOException, ConfigException, InterruptedException {
        super(config);

        homeWorkDatasyncClient = client("HomeWorkDataSync", config.homeWorkDatasyncConfig());
        deliveryDatasyncClient = client("DeliveryDataSync", config.deliveryDatasyncConfig());
        blackboxClient =
            registerClient(
                "Blackbox",
                new BlackboxClient(reactor, config.blackboxConfig()),
                config.blackboxConfig());
        geocoderClient = new AsyncGeocoderClient(this);

        Vertx vertx = Vertx.vertx();
        closeChain.add(vertx::close);
        pgClient = new PgClient(vertx, config.passportPgClientConfig(), logger);
        closeChain.add(pgClient);

        RegionServiceWrapper.init(this);

        Map<AddressService, AddressStorage<AddressBuilder>> storages = new LinkedHashMap<>();
        storages.put(AddressService.MAPS, new HomeWorkDataSyncStorage(this));
        storages.put(AddressService.MARKET, new MarketPgStorage(this));
        storages.put(AddressService.PASSPORT_DELIVERY, new PassportDeliveryDataSyncStorage(this));

        Set<AddressService> passportServices = new LinkedHashSet<>();
        for (AddressService service: AddressService.values()) {
            if (service == AddressService.UNKNOWN) {
                continue;
            }

            if (!storages.containsKey(service)) {
                passportServices.add(service);
            }
        }

        this.passportServices = Collections.unmodifiableSet(passportServices);

        try {
            Map<AddressService, List<AddressService>> passportServicesMap = new LinkedHashMap<>();
            for (AddressService service: passportServices) {
                Map<AddressService, ReadWriteRights> map =
                    AddressServicesRights.INSTANCE.allows().getOrDefault(service, Collections.emptyMap());
                if (map.isEmpty()) {
                    logger.warning("Empty allowed storages for " + service);
                }
                for (Map.Entry<AddressService, ReadWriteRights> entry : map.entrySet()) {
                    if (entry.getValue() == ReadWriteRights.NONE) {
                        continue;
                    }

                    passportServicesMap.computeIfAbsent(service, (k)-> new ArrayList<>()).add(entry.getKey());
                }
            }
            PassportPgStorage passportPgStorage = new PassportPgStorage(this, passportServices, passportServicesMap);
            for (AddressService service: passportServices) {
                storages.put(service, passportPgStorage);
            }

        } catch (ConfigException ce) {
            throw new IOException(ce);
        }
        this.addressStorages = Collections.unmodifiableMap(storages);

        Map<AddressService, Set<AddressStorage<AddressBuilder>>> allowedReads = new LinkedHashMap<>();
        Map<AddressService, Set<AddressStorage<AddressBuilder>>> allowedWrites = new LinkedHashMap<>();
        for (AddressService service: AddressService.values()) {

            Set<AddressStorage<AddressBuilder>> reads = new LinkedHashSet<>();
            Set<AddressStorage<AddressBuilder>> writes = new LinkedHashSet<>();
            Map<AddressService, ReadWriteRights> map =
                AddressServicesRights.INSTANCE.allows().getOrDefault(service, Collections.emptyMap());
            if (map.isEmpty()) {
                logger.warning("Empty allowed storages for " + service);
            }
            for (Map.Entry<AddressService, ReadWriteRights> entry : map.entrySet()) {
                if (entry.getValue() == ReadWriteRights.NONE) {
                    continue;
                }

                reads.add(this.addressStorages.get(entry.getKey()));
                if (entry.getValue() == ReadWriteRights.READ_WRITE) {
                    writes.add(this.addressStorages.get(entry.getKey()));
                }
            }

            allowedReads.put(service, Collections.unmodifiableSet(reads));
            allowedWrites.put(service, Collections.unmodifiableSet(writes));
        }

        this.allowedReads = Collections.unmodifiableMap(allowedReads);
        this.allowedWrites = Collections.unmodifiableMap(allowedWrites);

        this.register(
            new Pattern<>("/address/list", false),
            new ListAddressHandler(this));
        this.register(
            new Pattern<>("/address/get", false),
            new GetAddressHandler(this));
        this.register(
            new Pattern<>("/address/update", false),
            new UpdateAddressHandler(this));
        this.register(
            new Pattern<>("/address/create", false),
            new CreateAddressHandler(this));
        this.register(
            new Pattern<>("/address/delete", false),
            new DeleteAddressHandler(this));

        this.register(
            new Pattern<>("/contact/create", false),
            new CreateContactHandler(this));
        this.register(
            new Pattern<>("/contact/list", false),
            new ListContactHandler(this));
        this.register(
            new Pattern<>("/contact/get", false),
            new GetContactHandler(this));
        this.register(
            new Pattern<>("/contact/update", false),
            new UpdateContactHandler(this));
        this.register(
            new Pattern<>("/contact/delete", false),
            new DeleteContactHandler(this));

//        this.register(
//            new Pattern<>("/api/async/passport/multisearch", false),
//            new PassportMultidataSearchHandler(this, null));

    }

    public BlackboxClient blackboxClient() {
        return blackboxClient;
    }

    public AsyncGeocoderClient geocoderClient() {
        return geocoderClient;
    }

    public AsyncClient homeWorkDatasyncClient() {
        return homeWorkDatasyncClient;
    }

    public AsyncClient deliveryDatasyncClient() {
        return deliveryDatasyncClient;
    }

    public DBConnectionPool pool(final ImmutableAddressPgConfig config) {
        DBConnectionPool pool = pools.get(config.url());
        if (pool == null) {
            DBConnectionPool newPool =
                new DBConnectionPool(config.url(), config, logger);
            pool = pools.putIfAbsent(config.url(), newPool);
            if (pool == null) {
                pool = newPool;
                logger.info("POOL_CREATED " + config.url());
            }
        }

        return pool;
    }

    public PgClient pgClient() {
        return pgClient;
    }

    public void addTvmClient(final String tvmId) {
        tvm2ClientTicketGenerator.addDestId(tvmId);
    }

    public String ticket(final String tvmId) {
        return tvm2ClientTicketGenerator.ticket(tvmId);
    }

    public AddressStorage<AddressBuilder> storage(final AddressService service) {
        return addressStorages.get(service);
    }

    public Collection<AddressStorage<AddressBuilder>> allowedReads(final AddressService service) {
        return allowedReads.get(service);
    }

    public Collection<AddressStorage<AddressBuilder>> allowedWrites(final AddressService service) {
        return allowedWrites.get(service);
    }

    public Set<AddressService> passportServices() {
        return passportServices;
    }
}
