package ru.yandex.passport.address;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.CodingErrorAction;
import java.sql.SQLException;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;

public abstract class AbstractHandler implements ProxyRequestHandler {
    protected static String buildStmt(final Map<String, String> columns) {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, String> entry: columns.entrySet()) {
            if (entry.getValue().equals(entry.getKey())) {
                sb.append(entry.getKey());
            } else {
                sb.append(entry.getValue());
                sb.append(" as ");
                sb.append(entry.getKey());
            }

            sb.append(',');
        }

        sb.setLength(sb.length() - 1);

        return sb.toString();
    }

    protected void writeErrorResponse(
        final ProxySession session,
        final int status,
        final String error)
        throws IOException
    {
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
            writer.startObject();
            writer.key("status");
            writer.value("error");
            writer.key("error");
            writer.value(error);
            writer.endObject();
        }
        session.response(
            status,
            new NStringEntity(
                sbw.toString(),
                ContentType.APPLICATION_JSON
                    .withCharset(session.acceptedCharset())));
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        try {
            handleInternal(session);
        } catch (BadRequestException bre) {
            session.logger().log(Level.WARNING, "Request failed with exception: ", bre);
            writeErrorResponse(session, HttpStatus.SC_BAD_REQUEST, bre.getMessage());
        } catch (Exception e) {
            session.logger().log(Level.WARNING, "Request failed with exception: ", e);
            writeErrorResponse(session, HttpStatus.SC_INTERNAL_SERVER_ERROR, e.getMessage());
        }
    }

    protected OutputStreamWriter outWriter(
        final ProxySession session,
        final DecodableByteArrayOutputStream stream)
        throws HttpException
    {
        return
            new OutputStreamWriter(
                stream,
                CharsetUtils.acceptedCharset(session.request())
                    .newEncoder()
                    .onMalformedInput(
                        CodingErrorAction.REPLACE)
                    .onUnmappableCharacter(
                        CodingErrorAction.REPLACE));
    }

    public abstract void handleInternal(
        final ProxySession session)
        throws HttpException, IOException, SQLException, JsonException;
}
