package ru.yandex.passport.address;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import java.util.UUID;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;

import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.passport.AddressServiceParser;
import ru.yandex.util.string.HexStrings;

public class AddressId implements JsonValue {
    public static final char SEPARATOR = '/';

    private final Object userId;
    private final String userType;
    private final AddressService ownerService;
    private final String addressId;
    private final boolean serviceOriginalId;

    public AddressId(
        final String userType,
        final Object userId,
        final AddressService ownerService,
        final String addressId,
        final boolean serviceOriginalId)
    {
        this.userId = userId;
        this.userType = userType;
        this.ownerService = ownerService;
        this.addressId = addressId;
        this.serviceOriginalId = serviceOriginalId;
    }

    public AddressId(
        final String userType,
        final Object userId,
        final AddressService ownerService,
        final String addressId)
    {
        this(userType, userId, ownerService, addressId, false);
    }

    public AddressId(final String userType, final Object userId, final AddressService ownerService) {
        this.userId = userId;
        this.userType = userType;
        this.ownerService = ownerService;
        this.addressId = UUID.randomUUID().toString();
        this.serviceOriginalId = false;
    }

    public Object userId() {
        return userId;
    }

    public String userIdString() {
        return userId.toString();
    }

    public String userType() {
        return userType;
    }

    public AddressService ownerService() {
        return ownerService;
    }

    public String addressId() {
        return addressId;
    }

    public static String toString(
        final String userType,
        final Object userId,
        final String ownerService,
        final String addressId)
    {
        StringBuilder sb = new StringBuilder(addressId.length() + 40);
        sb.append(userType);
        sb.append(SEPARATOR);
        sb.append(userId);
        sb.append(SEPARATOR);
        sb.append(ownerService);
        sb.append(SEPARATOR);
        sb.append(addressId);
        return sb.toString();
    }

    public static String makeOuterId(
        final String userType,
        final Object userId,
        final AddressService ownerService,
        final String uuid,
        final boolean taxiOriginalId)
    {
        if (taxiOriginalId) {
            if (uuid.endsWith("-home")) {
                return "home";
            }

            if (uuid.endsWith("-work")) {
                return "work";
            }

            return uuid;
        }

        StringBuilder sb = new StringBuilder();
        byte[] userTypeBytes = userType.getBytes(StandardCharsets.UTF_8);
        HexStrings.LOWER.toStringBuilder(sb, userTypeBytes);
        sb.append('-');

        if (userId instanceof Number) {
            sb.append(userId);
        } else {
            byte[] userIdBytes = userId.toString().getBytes(StandardCharsets.UTF_8);
            HexStrings.LOWER.toStringBuilder(sb, userIdBytes);
        }

        sb.append('-');

        byte[] ownerServiceBytes = ownerService.serviceName().getBytes(StandardCharsets.UTF_8);
        HexStrings.LOWER.toStringBuilder(sb, ownerServiceBytes);
        sb.append('-');
        sb.append(uuid);
        return sb.toString();
    }

    public static AddressId parseOuterId(final String id) throws IOException {
        try {
            int index = id.indexOf('-');
            if (index < 0 || index >= id.length() - 1) {
                throw new IOException("Invalid id format " + id);
            }
            String userType = new String(Hex.decodeHex(id.substring(0, index)), StandardCharsets.UTF_8);
            int start = index + 1;
            index = id.indexOf('-', start);
            if (index >= id.length() - 1) {
                throw new IOException("Invalid id format " + id);
            }
            Object userId;
            if ("uid".equalsIgnoreCase(userType)) {
                userId = Long.parseLong(id.substring(start, index));
            } else {
                userId = new String(Hex.decodeHex(id.substring(start, index)), StandardCharsets.UTF_8);
            }

            start = index + 1;
            index = id.indexOf('-', start);
            if (index >= id.length() - 1) {
                throw new IOException("Invalid id format " + id);
            }
            String ownerServiceName = new String(Hex.decodeHex(id.substring(start, index)), StandardCharsets.UTF_8);
            AddressService ownerService = AddressServiceParser.INSTANCE.apply(ownerServiceName);
            String uuid = id.substring(index + 1);

            return new AddressId(userType, userId, ownerService, uuid);
        } catch (DecoderException de) {
            throw new IOException("Invalid id fornat " + de.getMessage(), de);
        }
    }

    @Override
    public String toString() {
        return makeOuterId(userType, userId, ownerService, addressId, serviceOriginalId);
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.value(makeOuterId(userType, userId, ownerService, addressId, serviceOriginalId));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || !(o instanceof AddressId)) {
            return false;
        }
        AddressId addressId1 = (AddressId) o;
        return serviceOriginalId == addressId1.serviceOriginalId
                   && userId.equals(addressId1.userId)
                   && userType.equals(addressId1.userType)
                   && ownerService == addressId1.ownerService
                   && addressId.equals(addressId1.addressId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(userId, userType, ownerService, addressId, serviceOriginalId);
    }
}
