package ru.yandex.passport.address;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Future;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;
import org.apache.http.nio.protocol.HttpAsyncResponseConsumer;
import org.apache.http.protocol.HttpContext;
import org.jetbrains.annotations.NotNull;

import ru.yandex.common.util.URLUtils;
import ru.yandex.common.util.geocoder.Component;
import ru.yandex.common.util.geocoder.GeoClient;
import ru.yandex.common.util.geocoder.GeoObject;
import ru.yandex.common.util.geocoder.GeoSearchParams;
import ru.yandex.common.util.geocoder.Kind;
import ru.yandex.common.util.geocoder.ProtoGeoObjectBuilder;
import ru.yandex.common.util.region.Region;
import ru.yandex.common.util.text.Charsets;
import ru.yandex.concurrent.CompletedFuture;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.AbstractAsyncByteArrayConsumer;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.passport.AddressProxy;
import ru.yandex.passport.RegionServiceWrapper;

import static ru.yandex.common.util.collections.CollectionUtils.filterList;
import static ru.yandex.common.util.collections.CollectionUtils.take;

public class AsyncGeocoderClient {
    private static final Comparator<Pair<Integer, GeoObject>> GEO_RESPONSE_COMPARATOR
        = Comparator.<Pair<Integer, GeoObject>>comparingInt(
        indexedGeoObject -> indexedGeoObject.getRight().getPrecision().getWeight()
    ).thenComparing(Pair::getLeft);

    private final AsyncClient client;
    private final ImmutableURIConfig config;
    private final ProxySession session;
    private final PrefixedLogger logger;

    private static final ProtoGeoObjectBuilder GEO_OBJECT_BUILDER
        = new ProtoGeoObjectBuilder();

    public AsyncGeocoderClient(final AddressProxy proxy) throws IOException {
        this.client = proxy.client("GeocoderProxy", proxy.config().geocoderConfig());
        this.config = proxy.config().geocoderConfig();
        this.session = null;
        this.logger = proxy.logger();
    }

    private AsyncGeocoderClient(
        final AsyncGeocoderClient client,
        final ProxySession session)
    {
        this.client = client.client;
        this.config = client.config;
        this.session = session;
        this.logger = session.logger();
    }

    public AsyncGeocoderClient adjust(final ProxySession session) {
        return new AsyncGeocoderClient(this, session);
    }

    public Future<List<GeoObject>> find(
        final String query,
        final FutureCallback<List<GeoObject>> callback) {
        return find(query, GeoSearchParams.DEFAULT, callback);
    }

    public Future<List<GeoObject>> find(
        final String query,
        final GeoSearchParams params,
        final FutureCallback<List<GeoObject>> callback)
    {
        return find(query, params, GeoClient.QueryHints.nothing(), callback);
    }

    @Nonnull
    public Future<List<GeoObject>> find(
        final String query,
        final GeoSearchParams params,
        final GeoClient.QueryHints hints,
        final FutureCallback<List<GeoObject>> callback)
    {
        if (StringUtils.isBlank(query)) {
            callback.completed(Collections.emptyList());
            return new CompletedFuture<>(Collections.emptyList());
        }

        HttpHost host = config.host();
        String url = prepareUrl(config.uri().toString() + "&text=", query, params).toString();

        Supplier<? extends HttpClientContext> contextGenerator;
        if (session != null) {
            contextGenerator = session.listener().createContextGeneratorFor(client);
        } else {
            contextGenerator = client.httpClientContextGenerator();
        }

        this.logger.fine("Sending geocoder request " + host + " " + url);
        return client.execute(
            host,
            new BasicAsyncRequestProducerGenerator(url),
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                new GeoCoderConsumerFactory(params)),
            contextGenerator,
            callback);
    }

    public void fillAddress(
        final PassportAddressBuilder address,
        final FutureCallback<? super PassportAddressBuilder> callback)
    {
        find(constructAddressQuery(address), new MergeCallback(address, callback));
    }

    public void fillAddress(
        final PassportAddressBuilder address,
        final GeoSearchParams params,
        final FutureCallback<? super PassportAddressBuilder> callback)
    {
        find(constructAddressQuery(address), params, new MergeCallback(address, callback));
    }

    private class MergeCallback extends AbstractFilterFutureCallback<List<GeoObject>, PassportAddressBuilder> {
        private final PassportAddressBuilder builder;

        public MergeCallback(
            final PassportAddressBuilder builder,
            final FutureCallback<? super PassportAddressBuilder> callback)
        {
            super(callback);
            this.builder = builder;
        }

        @Override
        public void completed(final List<GeoObject> geoObjects) {
            if (geoObjects == null || geoObjects.size() == 0) {
                callback.completed(builder);
                return;
            }

            PassportAddressBuilder fromGeo;
            if (geoObjects.size() == 1) {
                fromGeo = addressFromGeocoder(geoObjects.get(0), builder);
            } else {
                GeoObject geoObject = IntStream.range(0, geoObjects.size())
                    .mapToObj(i -> Pair.of(i, geoObjects.get(i)))
                    .min(GEO_RESPONSE_COMPARATOR)
                    .map(Pair::getRight)
                    .orElse(null);

                if (geoObject == null) {
                    callback.completed(builder);
                    return;
                }

                fromGeo = addressFromGeocoder(geoObject, builder);
            }

            callback.completed(fromGeo);
            return;
        }
    }

    public static @NotNull Integer tryWorkaroundNullRegionId(@Nullable String city, @NotNull Function<String,
        List<Region>> supplier) throws IOException {
        if (city == null) {
            throw new IOException("null city");
        }
        final List<Region> found = supplier.apply(city);
        if (found.size() == 1) {
            return found.get(0).getId();
        } else {
            throw new IOException(String.format("Several regions matches city name %s: %s", city,
                found.stream().map(Region::getName).collect(Collectors.toSet())));
        }
    }

    private static Location parseLocationFromGeocoder(final String point) {
        if (point == null) {
            return null;
        }
        String pointStr = point.strip();
        int index = pointStr.indexOf(' ');
        if (index > 0 && index < pointStr.length() - 1) {
            BigDecimal longitude = new BigDecimal(pointStr.substring(0, index));
            BigDecimal latitude = new BigDecimal(pointStr.substring(index + 1));
            return new Location(latitude, longitude);
        } else {
            return null;
        }
    }

//    private String getFullRegionPath(int input) {
//        RegionTree<? extends Region> regionTree = regionService.getRegionTree();
//        List<Integer> pathToRoot = new ArrayList<>(regionTree.getPathToRoot(input));
//        Collections.reverse(pathToRoot);
//        return pathToRoot.stream()
//            .map(regionTree::getRegion)
//            .filter(Objects::nonNull)
//            .map(Region::getName)
//            .collect(Collectors.joining(" "));
//    }

    private PassportAddressBuilder addressFromGeocoder(
        final GeoObject geoObject,
        final PassportAddressBuilder builder)
    {
        logger.info("Got from geocoder " + geoObject);
        GeoCoderPoint point = new GeoCoderPoint(geoObject.getPoint());
        String locality = null;
        for (Component component: geoObject.getComponents()) {
            if (component.getKinds().contains(Kind.LOCALITY)) {
                if (StringUtils.isNotBlank(component.getName())) {
                    locality = component.getName();
                    break;
                }
            }
        }
        Integer regionId = Optional.ofNullable(geoObject.getGeoid())
            .filter(StringUtils::isNotBlank)
            .filter(StringUtils::isNumeric)
            .map(Integer::valueOf)
            .orElse(null);
        PassportAddressBuilder result = builder
            .setCountry(geoObject.getCountryName())
            .setRegionId(regionId)
            .setRegion(RegionServiceWrapper.findRegionName(regionId))
            .setZip(geoObject.getPostalCode())
            .setBuilding(geoObject.getPremiseNumber())
            .setGeoCoderPoint(point);

        if (!geoObject.getThoroughfareName().isEmpty()) {
            result = builder.setStreet(geoObject.getThoroughfareName());
        }

        if (!geoObject.getDependentLocalityName().isEmpty()) {
            result = builder.setDistrict(geoObject.getDependentLocalityName());
        }

        if (locality != null) {
            builder.setCity(locality);
        } else {
            builder.setCity(
                Stream.of(
                        geoObject.getAdministrativeAreaName(),
                        geoObject.getSubAdministrativeAreaName()
                    )
                    .filter(org.apache.commons.lang3.StringUtils::isNotBlank)
                    .collect(Collectors.joining(", ")));
        }

        if (builder.location() == null) {
            builder.setLocation(parseLocationFromGeocoder(geoObject.getPoint()));
        }
        if (result.preciseRegionId() == null && result.regionId() != null) {
            result.setPreciseRegionId(result.regionId());
        }

        builder.setMarketAddressLine(geoObject.getAddressLine());

        return result;
    }

    private @NotNull String constructAddressQuery(final Address address) {
        if (StringUtils.isNotBlank(address.addressLine())) {
            return address.addressLine();
        } else {
            StringBuilder query = new StringBuilder();
            if (address.zip() != null) {
                query.append(address.zip());
                query.append(", ");
            }

            query.append(address.country());
            query.append(", ");
            if (address.regionId() != null) {
                query.append(address.regionId());
                query.append(", ");
            }

            query.append(address.city());
            query.append(", ");
            if (address.street() == null || address.street().isEmpty()) {
                query.append(address.district());
            } else {
                query.append(address.street());
            }
            query.append(", ");

            if (address.km() != null) {
                query.append(address.km());
                query.append(" километр ");
            }

            query.append(address.building());

            if (address.block() != null) {
                query.append(" строение ");
                query.append(address.block());
            }

            if (address.estate() != null) {
                query.append(" владение ");
                query.append(address.estate());
            }

            if (address.wing() != null) {
                query.append(" корпус ");
                query.append(address.wing());
            }

            return query.toString();
        }
    }

    private static CharSequence prepareUrl(
        final String baseUrl,
        final String query,
        final GeoSearchParams params)
    {
        final StringBuilder endUrl = new StringBuilder(1024).append(baseUrl).append(URLUtils.encode(query, Charsets.UTF_8));
        endUrl.append("&lang=").append(params.getPreferredLanguage().getLocale()); //geocoder only supports lang now.
        endUrl.append("&ms=pb");  // new geosearch api required parameter - proto format
        endUrl.append("&type=geo");  // new geosearch api parameter - output geocoder meta info

        if (params.getRegionId() != null) {
            endUrl.append(String.format("&lr=%d", params.getRegionId()));
        }

        if (params.getSearchAreaCenter() != null) {
            GeoSearchParams.Coordinates searchAreaCenter = params.getSearchAreaCenter();
            endUrl.append(String.format(
                "&ll=%f,%f",
                searchAreaCenter.getLongitude(),
                searchAreaCenter.getLatitude()
            ));
        }

        if (params.getSearchAreaSize() != null) {
            GeoSearchParams.SearchAreaSize searchAreaSize = params.getSearchAreaSize();
            endUrl.append(String.format(
                "&spn=%f,%f",
                searchAreaSize.getLength(),
                searchAreaSize.getWidth()
            ));
        }

        if (params.isSearchAreaLimited()) {
            endUrl.append("&rspn=1");
        }

        final Map<String, String> additionalParams = params.getAdditionalParams();
        if (additionalParams != null && additionalParams.size() > 0) {
            for (Map.Entry<String, String> entry : additionalParams.entrySet()) {
                endUrl.append("&").append(entry.getKey()).append("=").append(entry.getValue());
            }
        }
        return endUrl;
    }

    protected final List<GeoObject> prepareGeoObjects(
        final GeoSearchParams params,
        final byte[] s)
    {
        if (s.length == 0) {
            return Collections.emptyList();
        }

        final List<GeoObject> nonFilteredObjects = GEO_OBJECT_BUILDER.buildsGeoObjects(s);

        // returning first params.getLimit() of filtered by precision
        return take(params.getLimit(), filterList(nonFilteredObjects, geoObject ->
            geoObject.getPrecision().getWeight() <= params.getMinimalPrecision().getWeight()));
    }

    private class GeoCoderConsumerFactory
        implements HttpAsyncResponseConsumerFactory<List<GeoObject>>
    {
        private final GeoSearchParams params;

        public GeoCoderConsumerFactory(final GeoSearchParams params) {
            this.params = params;
        }

        @Override
        public HttpAsyncResponseConsumer<List<GeoObject>> create(
            final HttpAsyncRequestProducer producer,
            final HttpResponse response)
            throws HttpException, IOException
        {
            return new GeocoderConsumer(params);
        }
    }

    private class GeocoderConsumer
        extends AbstractAsyncByteArrayConsumer<List<GeoObject>>
    {
        private final GeoSearchParams params;

        public GeocoderConsumer(final GeoSearchParams params) {
            this.params = params;
        }

        @Override
        protected List<GeoObject> buildResult(final HttpContext context) throws Exception {
            if (buf == null) {
                return Collections.emptyList();
            } else {
                return prepareGeoObjects(params, Arrays.copyOf(buf, len));
            }
        }
    }

}
