package ru.yandex.passport.address;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.common.util.geocoder.GeoSearchParams;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.passport.AddressContext;
import ru.yandex.passport.AddressProxy;
import ru.yandex.passport.storage.PassportPgStorage;
import ru.yandex.search.msal.pool.DBConnectionPool;

import static ru.yandex.common.util.geocoder.GeoSearchParams.Builder;

public class GeocoderAddressConverter {
    private static final String INSERT_ADDRESS =
        "INSERT INTO passport_address(user_type, user_id, object_key, region_id, precise_region_id, " +
            "country, city, street, building, floor, " +
            "room, entrance, intercom, zip, source, comment, latitude, longitude, district, " +
            "platform, owner_service, subtype, format_version, label, brand_name, phone_id, comment_courier, " +
            "geocoder_name, geocoder_exact, geocoder_description, geocoder_object_type, org_id, " +
            "name, uri, deleted, locale, source_object_key) " +
            "VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, " +
                "?, ? ,?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?) " +
            "RETURNING *";

    private final AddressProxy proxy;
    private final DBConnectionPool passportConnectionPool;

    public GeocoderAddressConverter(final AddressProxy proxy) {
        this.proxy = proxy;
        this.passportConnectionPool = proxy.pool(proxy.config().passportDbConfig());
    }

    public void convert(
        final AddressContext context,
        final FutureCallback<AddressBuilder> callback,
        final AddressId id)
        throws SQLException
    {
        ProxySession session = context.session();
        try (Connection connection = passportConnectionPool.getConnection(session.logger());
            PreparedStatement stmt = connection.prepareStatement(PassportPgStorage.GET_PASSPORT_ADDRESS))
        {
            stmt.setString(1, id.addressId());
            stmt.setString(2, Locale.RU.toString());
            stmt.setString(3, id.userIdString());
            stmt.setString(4, id.userType());

            session.logger().info(stmt.toString());
            try (ResultSet resultSet = stmt.executeQuery()) {
                if (!resultSet.next()) {
                    callback.failed(new ServerException(HttpStatus.SC_NOT_FOUND, "not found"));
                } else {
                    PassportAddressBuilder address = PassportAddressBuilder.build(context.userType(), context.userId(), resultSet);
                    if (!address.validForGeocoder()) {
                        context.session().logger().info(
                            "Not specified address_line or country,city,street,building in ru address");
                        callback.failed(new ServerException(HttpStatus.SC_NOT_FOUND, "not found"));
                    } else {
                        convertAddress(context, callback, address, context.service());
                    }
                }
            }
        }
    }

    public void convertList(
        final ListAddressContext context,
        final FutureCallback<List<AddressBuilder>> callback,
        final AddressService service)
        throws SQLException
    {
        ProxySession session = context.session();
        try (Connection connection = passportConnectionPool.getConnection(session.logger());
             PreparedStatement stmt = connection.prepareStatement(PassportPgStorage.PASSPORT_LIST_ADDRESSES))
        {
            stmt.setString(1, String.valueOf(context.userId()));
            stmt.setString(2, context.userType());
            stmt.setString(3, service.serviceName());
            stmt.setString(4, Locale.RU.toString());
            stmt.setInt(5, context.length());
            session.logger().info(stmt.toString());

            MultiFutureCallback<AddressBuilder> multiFutureCallback =
                new MultiFutureCallback<>(callback);

            try (ResultSet resultSet = stmt.executeQuery()) {
                while (resultSet.next()) {
                    PassportAddressBuilder address = PassportAddressBuilder.build(context.passportUser(), resultSet);
                    if (!address.validForGeocoder()) {
                        context.session().logger().info(
                            "Not specified address_line or country,city,street,building in ru address for "
                                + address.id().addressId() + ", ignore");
                    } else {
                        convertAddress(context, multiFutureCallback.newCallback(), address, service);
                    }
                }
            }
            multiFutureCallback.done();
        }
    }

    private void convertAddress(
        final AddressContext context,
        final FutureCallback<AddressBuilder> callback,
        final PassportAddressBuilder address,
        final AddressService service)
    {
        String sourceObjectKey = address.id().addressId();

        AddressId newId = new AddressId(context.userType(), context.userId(), service);
        address.setId(newId);

        AsyncGeocoderClient geocoderClient = proxy.geocoderClient().adjust(context.session());
        Builder paramBuilder = GeoSearchParams.builder()
            .withPreferredLanguage(context.locale().language());

        Location location = address.location();
        if (location != null && location.getLongitude() != null && location.getLatitude() != null) {
            paramBuilder = paramBuilder.withSearchAreaCenter(
                location.getLongitude().doubleValue(),
                location.getLatitude().doubleValue()
            );
        }

        GeoSearchParams params = paramBuilder.build();
        FillCallback fillCallback = new FillCallback(context, callback, sourceObjectKey);
        geocoderClient.fillAddress(address, params, fillCallback);
    }

    private class FillCallback extends AbstractProxySessionCallback<PassportAddressBuilder> {
        private final AddressContext context;
        private final FutureCallback<AddressBuilder> callback;
        private final String sourceObjectKey;

        public FillCallback(
            final AddressContext context,
            final FutureCallback<AddressBuilder> callback,
            final String sourceObjectKey)
        {
            super(context.session());
            this.context = context;
            this.callback = callback;
            this.sourceObjectKey = sourceObjectKey;
        }

        @Override
        public void completed(final PassportAddressBuilder address) {
            try (Connection connection = passportConnectionPool.getConnection(context.session().logger());
                PreparedStatement stmt = connection.prepareStatement(INSERT_ADDRESS))
            {
                stmt.setString(1, context.userType());
                stmt.setLong(2, context.userIdLong());
                stmt.setString(3, address.id().addressId());
                stmt.setObject(4, address.regionId());
                stmt.setObject(5, address.preciseRegionId());
                stmt.setString(6, address.country());
                stmt.setString(7, address.city());
                stmt.setString(8, address.street());
                stmt.setString(9, address.building());
                stmt.setString(10, address.floor());
                stmt.setString(11, address.room());
                stmt.setString(12, address.entrance());
                stmt.setString(13, address.intercom());
                stmt.setString(14, address.zip());
                stmt.setString(15, address.subtype());
                stmt.setString(16, address.comment());
                if (address.location() != null && address.location().getLatitude() != null) {
                    stmt.setObject(17, address.location().getLatitude().doubleValue());
                } else {
                    stmt.setObject(17, null);
                }
                if (address.location() != null && address.location().getLongitude() != null) {
                    stmt.setObject(18, address.location().getLongitude().doubleValue());
                } else {
                    stmt.setObject(18, null);
                }
                stmt.setString(19, address.district());
                stmt.setString(20, address.platform());
                stmt.setString(21, address.ownerService().serviceName());
                stmt.setString(22, address.subtype());

                stmt.setString(23, address.formatVersion());
                stmt.setString(24, address.label());
                stmt.setString(25, address.brandName());
                stmt.setString(26, address.phoneId());
                stmt.setString(27, address.commentCourier());
                stmt.setString(28, address.geocoderName());
                stmt.setObject(29, address.geocoderExact());
                stmt.setString(30, address.geocoderDescription());
                stmt.setString(31, address.geocoderObjectType());
                stmt.setObject(32, address.orgId());
                stmt.setString(33, address.name());
                stmt.setString(34, address.uri());
                stmt.setBoolean(35, false);
                stmt.setString(36, context.locale().toString());
                stmt.setString(37, sourceObjectKey);


                context.session().logger().info(stmt.toString());
                context.session().logger().info(address.toString());

                try (ResultSet resultSet = stmt.executeQuery()) {
                    if (!resultSet.next()) {
                        callback.failed(
                            new ServerException(
                                HttpStatus.SC_INTERNAL_SERVER_ERROR,
                                "Insertion returned empty response"));
                        return;
                    }
                    long id = resultSet.getLong("id");
                    context.session().logger().info("Saved " + id);
                    callback.completed(PassportAddressBuilder.build(context.userType(), context.userId(), resultSet));
                }
            } catch (Exception e) {
                callback.failed(e);
            }
        }
    }
}
