package ru.yandex.passport.address;

import java.util.Arrays;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

public enum GeocoderStatus {
    /**
     * Наличие координат не проверено.
     */
    INIT("INIT"),

    /**
     * Успешно полученные координаты с новым запросом.
     */
    SUCCESS_CONFIRMED("SUCCESS_CONFIRMED"),

    /**
     * Координаты получены из геокодера c точностью NUMBER.
     * Для части координат использовался старый запрос (вместе с деревом регионов)
     * и результат для некоторых адресов неверный
     */
    @Deprecated
    SUCCESS_WITH_PRECISION_NUMBER("SUCCESS_WITH_PRECISION_NUMBER"),

    /**
     * Координаты получены из геокодера c точностью NUMBER.
     */
    GEOCODER_NUMBER_PRECISION("GEOCODER_NUMBER_PRECISION"),

    /**
     * Геокодер не вернул координаты.
     */
    NOT_FOUND("NOT_FOUND"),

    /**
     * Геокодер не вернул объектов с заданной точностью.
     */
    GEOCODER_LOW_PRECISION("GEOCODER_LOW_PRECISION"),

    /**
     * Координаты получены из запроса c фронта.
     */
    FROM_REQUEST("FROM_REQUEST"),
    /**
     * Координаты получены из запроса c фронта и с проверенным регионом.
     */
    FROM_REQUEST_CONFIRMED("FROM_REQUEST_CONFIRMED");

    private String code;
    private static final Map<String, GeocoderStatus> cache = createCache(values());

    GeocoderStatus(String code) {
        this.code = code;
    }

    public static GeocoderStatus findByCode(String code) {
        return cache.get(code);
    }

    public String getCode() {
        return code;
    }

    static <T extends GeocoderStatus> Map<String, T> createCache(T[] values) {
        return Arrays.stream(values)
            .collect(Collectors.toMap(
                GeocoderStatus::getCode,
                Function.identity()
            ));
    }
}

