package ru.yandex.passport.address;

import java.time.OffsetDateTime;
import java.util.Objects;

import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.Nullable;

public class ImmutableAddress implements Address {
    @Nullable
    protected final AddressId id;
    protected final AddressService ownerService;
    protected final Integer regionId;
    protected final Integer preciseRegionId;
    protected final String country;
    protected final String zip;
    protected final String city;
    protected final String metro;
    protected final String street;
    protected final String building;
    protected final String entrance;
    protected final String intercom;
    protected final String floor;
    protected final String room;
    protected final Boolean cargoLift;
    protected final String addressLine;
    protected final String block;
    protected final String estate;
    protected final String wing;
    protected final String km;
    protected final String region;
    protected final String type;
    @Nullable
    protected final GeoCoderPoint geoCoderPoint;
    @Nullable
    protected final String comment;
    @Nullable
    protected final Location location;
    protected final GeocoderStatus geocoderStatus;
    @Nullable
    protected final String district;
    @Nullable
    protected final OffsetDateTime lastTouchedTime;
    @Nullable
    protected final String platform;
    protected final String fullnessState;

    public ImmutableAddress(
        @Nullable AddressId id,
        AddressService ownerService,
        Integer regionId,
        Integer preciseRegionId,
        String country,
        String zip,
        String city,
        String metro,
        String street,
        String building,
        String entrance,
        String intercom,
        String floor,
        String room,
        Boolean cargoLift,
        String block,
        String estate,
        String wing,
        String km,
        String region,
        String type,
        String addressLine,
        @Nullable GeoCoderPoint geoCoderPoint,
        @Nullable String comment,
        @Nullable Location location,
        GeocoderStatus geocoderStatus,
        @Nullable String district,
        @Nullable OffsetDateTime lastTouchedTime,
        @Nullable String platform,
        String fullnessState)
    {
        this.id = id;
        this.ownerService = ownerService;
        this.regionId = regionId;
        this.preciseRegionId = preciseRegionId;
        this.country = country;
        this.zip = zip;
        this.city = city;
        this.metro = metro;
        this.street = street;
        this.building = building;
        this.entrance = entrance;
        this.intercom = intercom;
        this.floor = floor;
        this.room = room;
        this.cargoLift = cargoLift;
        if (addressLine != null) {
            this.addressLine = addressLine;
        } else {
            StringBuilder sb = new StringBuilder();
            if (StringUtils.isNotBlank(country)) {
                sb.append(country);
                sb.append(", ");
            }

            if (StringUtils.isNotBlank(region)) {
                sb.append(region);
                sb.append(", ");
            }

            if (StringUtils.isNotBlank(city)) {
                sb.append(city);
                sb.append(", ");
            }

            if (StringUtils.isNotBlank(street)) {
                sb.append(street);
                sb.append(", ");
            } else {
                if (StringUtils.isNotBlank(district)) {
                    sb.append(district);
                    sb.append(", ");
                }
            }

//            if (StringUtils.isNotBlank(km)) {
//                sb.append(street);
//                sb.append(" километр , ");
//            }

            if (StringUtils.isNotBlank(building)) {
                sb.append(building);
                sb.append(", ");
            }

//            if (StringUtils.isNotBlank(block)) {
//                sb.append("строение ");
//                sb.append(block);
//                sb.append(", ");
//            }
//
//            if (StringUtils.isNotBlank(estate)) {
//                sb.append("владение ");
//                sb.append(estate);
//                sb.append(", ");
//            }
//
//            if (StringUtils.isNotBlank(wing)) {
//                sb.append("корпус ");
//                sb.append(wing);
//                sb.append(", ");
//            }
            if (sb.length() > 1) {
                sb.setLength(sb.length() - 2);
                this.addressLine = sb.toString();
            } else {
                this.addressLine = null;
            }
        }

        this.block = block;
        this.estate = estate;
        this.wing = wing;
        this.km = km;
        this.region = region;
        this.type = type;
        this.geoCoderPoint = geoCoderPoint;
        this.comment = comment;
        this.location = location;
        this.geocoderStatus = geocoderStatus;
        this.district = district;
        this.lastTouchedTime = lastTouchedTime;
        this.platform = platform;
        this.fullnessState = fullnessState;
    }

    public static AddressBuilder copy(ImmutableAddress address) {
        return new AddressBuilder()
            .setId(address.id())
            .setOwnerService(address.ownerService())
            .setRegionId(address.regionId())
            .setPreciseRegionId(address.preciseRegionId())
            .setCountry(address.country())
            .setZip(address.zip())
            .setCity(address.city())
            .setMetro(address.metro())
            .setStreet(address.street())
            .setBuilding(address.building())
            .setEntrance(address.entrance())
            .setIntercom(address.intercom())
            .setFloor(address.floor())
            .setRoom(address.room())
            .setCargoLift(address.cargoLift())
            .setMarketAddressLine(address.addressLine())
            .setBlock(address.block())
            .setEstate(address.estate())
            .setWing(address.wing())
            .setKm(address.km())
            .setRegion(address.region())
            .setType(address.subtype())
            .setGeoCoderPoint(address.geoCoderPoint())
            .setComment(address.comment())
            .setLocation(address.location())
            .setGeocoderStatus(address.geocoderStatus())
            .setDistrict(address.district())
            .setLastTouchedTime(address.lastTouchedTime())
            .setPlatform(address.platform())
            .setFullnessState(address.fullnessState());
    }

    @Nullable
    @Override
    public AddressId id() {
        return id;
    }

    @Override
    public AddressService ownerService() {
        return ownerService;
    }

    @Nullable
    @Override
    public GeoCoderPoint geoCoderPoint() {
        return geoCoderPoint;
    }

    @Override
    public Boolean cargoLift() {
        return cargoLift;
    }

    @Override
    public Integer regionId() {
        return regionId;
    }

    @Override
    public Integer preciseRegionId() {
        return preciseRegionId;
    }

    @Override
    public String country() {
        return country;
    }

    @Override
    public String zip() {
        return zip;
    }

    @Override
    public String city() {
        return city;
    }

    @Override
    public String metro() {
        return metro;
    }

    @Override
    public String street() {
        return street;
    }

    @Override
    public String building() {
        return building;
    }

    @Override
    public String entrance() {
        return entrance;
    }

    @Override
    public String intercom() {
        return intercom;
    }

    @Override
    public String floor() {
        return floor;
    }

    @Override
    public String room() {
        return room;
    }

    @Override
    public String addressLine() {
        return addressLine;
    }

    @Override
    public String subtype() {
        return type;
    }

    @Override
    public String comment() {
        return comment;
    }

    /**
     * Вернуть номер корпуса.
     *
     * @return
     */
    @Override
    public String block() {
        return block;
    }

    /**
     * Вернуть номер владения.
     *
     * @return
     */
    @Override
    public String estate() {
        return estate;
    }

    /**
     * Вернуть номер строения.
     *
     * @return
     */
    @Override
    public String wing() {
        return wing;
    }

    /**
     * Вернуть километр.
     *
     * @return
     */
    @Override
    public String km() {
        return km;
    }

    /**
     * Вернуть область или регион.
     *
     * @return
     */
    @Override
    public String region() {
        return region;
    }

    /**
     * Вернуть географические координаты
     */
    @Override
    public Location location() {
        return location;
    }

    /**
     * Вернуть статус обновления координат геокодером
     *
     * @return
     */
    @Override
    public GeocoderStatus geocoderStatus() {
        return geocoderStatus;
    }


    /**
     * Вернуть район
     */
    @Override
    public String district() {
        return district;
    }

    /**
     * Вернуть время последнего взаимодействия пользователя с адресом
     */
    @Override
    public OffsetDateTime lastTouchedTime() {
        return lastTouchedTime;
    }

    /**
     * Вернуть платформу с которой пользователь создал или изменил адрес
     */
    @Override
    public String platform() {
        return platform;
    }

    /**
     * Вернуть статус заполненности адреса
     */
    @Override
    public String fullnessState() {
        return fullnessState;
    }

    @Override
    public String toString() {
        return "MarketAddress{" +
            "id='" + id + '\'' +//
            ", regionId=" + regionId +//
            ", preciseRegionId=" + preciseRegionId +//
            ", country='" + country + '\'' +//
            ", zip='" + zip + '\'' +//
            ", city='" + city + '\'' +//
            ", metro='" + metro + '\'' +//
            ", street='" + street + '\'' +//
            ", building='" + building + '\'' +//
            ", entrance='" + entrance + '\'' +//
            ", intercom='" + intercom + '\'' +//
            ", floor='" + floor + '\'' +//
            ", room='" + room + '\'' +//
            ", cargoLift=" + cargoLift +//
            ", addressLine='" + addressLine + '\'' +//
            ", block='" + block + '\'' +//
            ", estate='" + estate + '\'' +//
            ", wing='" + wing + '\'' +//
            ", km='" + km + '\'' +//
            ", region='" + region + '\'' +//
            ", type=" + type +//
            ", geoCoderPoint=" + geoCoderPoint +//
            ", comment=" + comment +//
            ", location=" + location +
            ", geocoderStatus=" + geocoderStatus +
            ", district=" + district +
            ", platform=" + platform +
            ", fullnessState=" + fullnessState +
            '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || !(o instanceof ImmutableAddress)) {
            return false;
        }

        ImmutableAddress address = (ImmutableAddress) o;

        if (Objects.equals(geoCoderPoint, address.geoCoderPoint)) {
            return Objects.equals(entrance, address.entrance) &&
                Objects.equals(intercom, address.intercom) &&
                Objects.equals(floor, address.floor) &&
                Objects.equals(room, address.room) &&
                Objects.equals(cargoLift, address.cargoLift);
        } else {
            return Objects.equals(regionId, address.regionId) &&
                Objects.equals(preciseRegionId, address.preciseRegionId) &&
                Objects.equals(addressLine, address.addressLine) &&
                Objects.equals(country, address.country) &&
                Objects.equals(zip, address.zip) &&
                Objects.equals(city, address.city) &&
                Objects.equals(metro, address.metro) &&
                Objects.equals(street, address.street) &&
                Objects.equals(building, address.building) &&
                Objects.equals(block, address.block) &&
                Objects.equals(estate, address.estate) &&
                Objects.equals(wing, address.wing) &&
                Objects.equals(km, address.km) &&
                Objects.equals(region, address.region) &&
                Objects.equals(entrance, address.entrance) &&
                Objects.equals(intercom, address.intercom) &&
                Objects.equals(floor, address.floor) &&
                Objects.equals(room, address.room) &&
                Objects.equals(type, address.type) &&
                Objects.equals(cargoLift, address.cargoLift);
        }
    }

    @Override
    public int hashCode() {
        if (geoCoderPoint != null) {
            return Objects.hash(geoCoderPoint, entrance, intercom, floor, room, cargoLift);
        } else {
            return Objects.hash(addressLine, regionId, preciseRegionId, country, zip, city, metro, street, block,
                estate, wing, km, region, building, entrance, intercom, floor, room, type, cargoLift);
        }
    }

    public static AddressBuilder builder() {
        return new AddressBuilder();
    }

    public ImmutableAddress mergeWithGeocoder(GeocoderResponse geocodeMarketAddress) {
        return builder()
            .setId(id())
            .setMarketAddressLine(addressLine())
            .setBuilding(building())
            .setCargoLift(cargoLift())
            .setCity(city())
            .setCountry(country())
            .setEntrance(entrance())
            .setFloor(floor())
            .setIntercom(intercom())
            .setMetro(metro())
            .setRegionId(regionId())
            .setPreciseRegionId(preciseRegionId())
            .setRoom(room())
            .setStreet(street())
            .setZip(geocodeMarketAddress.getZip())
            .setBlock(block())
            .setEstate(estate())
            .setWing(wing())
            .setKm(km())
            .setRegion(region())
            .setType(subtype())
            .setGeoCoderPoint(geocodeMarketAddress.getPoint())
            .setComment(comment())
            .setLocation(location())
            .setGeocoderStatus(geocoderStatus())
            .setDistrict(district())
            .setPlatform(platform())
            .setFullnessState(fullnessState())
            .build();
    }

    public ImmutableAddress mergeWithRegion(String region) {
        return builder()
            .setId(id())
            .setMarketAddressLine(addressLine())
            .setBuilding(building())
            .setCargoLift(cargoLift())
            .setCity(city())
            .setCountry(country())
            .setEntrance(entrance())
            .setFloor(floor())
            .setIntercom(intercom())
            .setMetro(metro())
            .setRegionId(regionId())
            .setPreciseRegionId(preciseRegionId())
            .setRoom(room())
            .setStreet(street())
            .setZip(zip())
            .setBlock(block())
            .setEstate(estate())
            .setWing(wing())
            .setKm(km())
            .setRegion(region)
            .setType(subtype())
            .setGeoCoderPoint(geoCoderPoint())
            .setComment(comment())
            .setLocation(location())
            .setGeocoderStatus(geocoderStatus())
            .setDistrict(district())
            .setPlatform(platform())
            .setFullnessState(fullnessState())
            .build();
    }
}
