package ru.yandex.passport.address;

import java.io.IOException;
import java.math.BigDecimal;
import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;

import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.passport.AddressContext;
import ru.yandex.passport.AddressServiceParser;

public class PassportAddressDto {
    private static final DateTimeFormatter TAXI_FORMATTER;
    static {
        DateTimeFormatterBuilder builder = new DateTimeFormatterBuilder();
        builder.appendPattern("uuuu-MM-dd'T'HH:mm:ss");
        builder.appendFraction(ChronoField.MILLI_OF_SECOND, 0, 3, true);
        builder.appendOffset("+HHmm", "+0000");
        TAXI_FORMATTER = builder.toFormatter();
    }

    public static void write(
        final PassportAddress address,
        final JsonWriterBase writer)
        throws IOException
    {
        AddressDto.write(address, writer);
        writer.key("address_type");
        writer.value(address.subtype());
        writer.key("last_touched_time");
        writer.value(address.lastTouchedTime());
        writer.key("address_line");
        writer.value(address.addressLine());
        writer.key("locale");
        writer.value(address.locale());
        writer.key("version");
        writer.value(address.version());
        writer.key("format_version");
        writer.value(address.formatVersion());
        writer.key("name");
        writer.value(address.name());
        writer.key("uri");
        writer.value(address.uri());
        writer.key("label");
        writer.value(address.label());
        writer.key("brand_name");
        writer.value(address.brandName());
        writer.key("phone_id");
        writer.value(address.phoneId());
        writer.key("geocoder_name");
        writer.value(address.geocoderName());
        writer.key("geocoder_description");
        writer.value(address.geocoderDescription());
        writer.key("geocoder_object_type");
        writer.value(address.geocoderObjectType());
        writer.key("geocoder_exact");
        writer.value(address.geocoderExact());
        writer.key("comment_courier");
        writer.value(address.commentCourier());
        writer.key("org_id");
        writer.value(address.orgId());
        writer.key("modification_time");
        writer.value(address.modificationTime());
        writer.key("creation_time");
        writer.value(address.creationTime());
        writer.key("draft");
        writer.value(address.draft());
        writer.key("is_portal_uid");
        writer.value(address.isPortalUid());
    }

    public static PassportAddressBuilder parseForCreate(
        final AddressContext context,
        final JsonMap result)
        throws JsonException
    {
        PassportAddressBuilder builder = new PassportAddressBuilder();
        builder.setMarketAddressLine(result.getString("address_line", null));
        builder.setRegionId(result.getInt("region_id", null));
        builder.setPreciseRegionId(builder.regionId());
        AddressService ownerService =
            result.get("owner_service", context.service(), AddressServiceParser.INSTANCE);
        builder.setOwnerService(ownerService);

        builder.setCountry(result.getString("country", null));
        builder.setZip(result.getString("zip", null));
        String city = result.getString("locality", null);
        if (city == null) {
            city = result.getString("city", null);
        }
        builder.setCity(city);
        String subtype = result.getString("address_type", null);
        if (subtype == null) {
            subtype = result.getString("subtype", "address");
        }
        builder.setType(subtype);
        builder.setStreet(result.getString("street", null));
        builder.setBuilding(result.getString("building", null));
        builder.setIntercom(result.getString("intercom", null));
        builder.setFloor(result.getString("floor", null));
        builder.setRoom(result.getString("room", null));
        builder.setDraft(result.getBoolean("draft", false));

        JsonMap locationMap = result.getMapOrNull("location");
        if (locationMap != null) {
            builder.setLocation(
                new Location(
                    BigDecimal.valueOf(locationMap.getDouble("latitude")),
                    BigDecimal.valueOf(locationMap.getDouble("longitude"))));
        } else {
            builder.setLocation(null);
        }

        builder.setDistrict(result.getString("district", null));
        builder.setComment(result.getString("comment", null));
        builder.setEntrance(result.getString("entrance", null));

        builder.setLocale(result.getString("locale", Locale.RU.toString()));
        builder.setVersion(result.getInt("version", 0));
        builder.setFormatVersion(result.getString("format_version", null));
        builder.setName(result.getString("name", null));
        builder.setUri(result.getString("uri", null));
        builder.setLabel(result.getString("label", null));
        builder.setBrandName(result.getString("brand_name", null));
        builder.setPhoneId(result.getString("phone_id", null));
        builder.setGeocoderName(result.getString("geocoder_name", null));
        builder.setGeocoderDescription(result.getString("geocoder_description", null));
        builder.setGeocoderObjectType(result.getString("geocoder_object_type", null));
        builder.setGeocoderExact(result.getBoolean("geocoder_exact", null));
        builder.setCommentCourier(result.getString("comment_courier", null));
        builder.setOrgId(result.getLong("org_id", null));
        String lastTouchedTime = result.getString("last_touched_time", null);
        builder.setLastTouchedTime(parseOffsetDateTime(lastTouchedTime));
        String modificationTime = result.getString("modification_time", null);
        builder.setModificationTime(parseOffsetDateTime(modificationTime));
        String creationTime = result.getString("creation_time", null);
        builder.setCreationTime(parseOffsetDateTime(creationTime));
        builder.setPortalUid(result.getBoolean("is_portal_uid", false));
        return builder;
    }

    private static OffsetDateTime parseOffsetDateTime(String field) {
        if (field == null) {
            return null;
        }

        try {
            return OffsetDateTime.parse(field);
        } catch (DateTimeParseException dtpe) {

            //DateTimeFormatter.ofPattern("uuuu-MM-dd'T'HH:mm:ss.[SSS][SS][S][]ZZZ")
            return OffsetDateTime.parse(field, TAXI_FORMATTER);
            //OffsetDateTime.parse(field, DateTimeFormatter.ISO_OFFSET_DATE_TIME)
        }
    }
}
