package ru.yandex.passport.address.handlers;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.CodingErrorAction;
import java.sql.SQLException;
import java.time.OffsetDateTime;
import java.util.Objects;
import java.util.logging.Level;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.passport.AbstractAddressSessionCallback;
import ru.yandex.passport.AddressContext;
import ru.yandex.passport.AddressProxy;
import ru.yandex.passport.AddressStorage;
import ru.yandex.passport.address.AddressBuilder;
import ru.yandex.passport.address.AddressId;
import ru.yandex.passport.address.AsyncGeocoderClient;
import ru.yandex.passport.address.PassportAddress;
import ru.yandex.passport.address.PassportAddressBuilder;
import ru.yandex.passport.address.PassportAddressDto;

public class CreateAddressHandler extends AbstractAddressHandler implements ProxyRequestHandler {
    private static final SqlQuery INSERT_ADDRESS_VERTX =  new SqlQuery(
        "create_address",
         "INSERT INTO passport_address(user_type, user_id, object_key, region_id, precise_region_id, " +
         "country, city, street, building, floor, " +
         "room, entrance, intercom, zip, source, comment, latitude, longitude, district, " +
         "platform, owner_service, subtype, deleted, draft, creation_time, modification_time, last_touched_time, locale) " +
         "VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $20, $21, $22, $23, $24, $25, $26 ,$27, $28) " +
         "RETURNING id");

    private static final SqlQuery INSERT_DRAFT_ADDRESS_VERTX =
        new SqlQuery(
            "insert_address_draft",
         "INSERT INTO passport_address(user_type, user_id, object_key, owner_service, subtype, deleted, draft, creation_time, modification_time, last_touched_time) " +
         "VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10) " +
         "RETURNING id");

    //private final DBConnectionPool passportConnectionPool;
    private final AddressProxy proxy;

    public CreateAddressHandler(final AddressProxy proxy) {
        //this.passportConnectionPool = proxy.pool(proxy.config().passportDbConfig());
        this.proxy = proxy;
    }

    @Override
    public void handleInternal(
        final ProxySession session)
        throws HttpException, IOException, SQLException, JsonException
    {
        final AddressContext context = new AddressContext(session);
        if (!(session.request() instanceof HttpEntityEnclosingRequest)) {
            writeErrorResponse(session, HttpStatus.SC_BAD_REQUEST, "No body supplied for request");
            return;
        }

        JsonMap addressDto =
            TypesafeValueContentHandler.parse(
                CharsetUtils.content(
                    ((HttpEntityEnclosingRequest) session.request()).getEntity())).asMap();
        PassportAddressBuilder address = PassportAddressDto.parseForCreate(context, addressDto);
        address.setId(new AddressId(context.userType(), context.userId(), address.ownerService()));

        if (!context.modifyAllowed(address.ownerService())) {
            writeErrorResponse(
                session,
                HttpStatus.SC_FORBIDDEN,
                "Service " + context.service() + " has not permission to modify or create in namespace " + address.ownerService());
            return;
        }

        boolean taxi = "taxi".equalsIgnoreCase(context.serviceName());
        if (!taxi && !address.validForGeocoder()) {
            writeErrorResponse(
                session,
                HttpStatus.SC_BAD_REQUEST,
                "address_line or country,city,street or district,building should be specified");
            return;
        }

        context.session().logger().info(JsonType.NORMAL.toString(addressDto));


        FillCallback callback = new FillCallback(context);
        if (context.serviceConfig().serviceKeepHisId()) {
            String id = session.params().getString("id", null);

            if (id != null) {
                address.setId(parseWithTaxi(context, id));
                //new AddressId(context.userType(), context.userId(), context.serviceName(), id));
            }

            Tuple stmt = Tuple.tuple();
            stmt.addString(context.userType());
            stmt.addString(String.valueOf(context.userId()));
            stmt.addString(address.id().addressId());
            stmt.addString(address.ownerService().serviceName());
            stmt.addString(address.subtype());
            stmt.addBoolean(false);
            stmt.addBoolean(context.draft());
            OffsetDateTime creationTime = getDateFieldOrNow(address.creationTime());
            stmt.addLocalDateTime(creationTime.toLocalDateTime());
            OffsetDateTime modificationTime = getDateFieldOrNow(address.modificationTime());
            stmt.addLocalDateTime(modificationTime.toLocalDateTime());
            if (address.lastTouchedTime() != null) {
                stmt.addLocalDateTime(address.lastTouchedTime().toLocalDateTime());
            } else {
                stmt.addLocalDateTime(null);
            }

            address.setCreationTime(creationTime);
            address.setModificationTime(modificationTime);

            context.session().logger().info(stmt.toString());
            proxy.pgClient().executeOnMaster(
                INSERT_DRAFT_ADDRESS_VERTX,
                stmt,
                context.session().listener(),
                new CreateCallback(callback, context, address));
        } else {
            AsyncGeocoderClient geocoderClient = proxy.geocoderClient().adjust(session);
            geocoderClient.fillAddress(address, callback);
        }
    }

    private class CreateCallback extends AbstractFilterFutureCallback<RowSet<Row>, PassportAddressBuilder> {
        private final AddressContext context;
        private final PassportAddressBuilder address;

        public CreateCallback(
            final FutureCallback<? super PassportAddressBuilder> callback,
            final AddressContext context,
            final PassportAddressBuilder address)
        {
            super(callback);
            this.context = context;
            this.address = address;
        }

        @Override
        public void completed(final RowSet<Row> rows) {
            try {
                if (rows.rowCount() <= 0) {
                    writeErrorResponse(context.session(), HttpStatus.SC_INTERNAL_SERVER_ERROR, "Update returned empty response");
                    return;
                }

                Row resultSet = rows.iterator().next();
                Long itemid = resultSet.getLong("id");
                context.session().logger().info("Saved " + itemid);

                writeResults(address, context);
            } catch (Exception e) {
                callback.failed(e);
            }
        }
    }
    private class FillCallback extends AbstractProxySessionCallback<PassportAddressBuilder> {
        private final AddressContext context;

        public FillCallback(final AddressContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final PassportAddressBuilder address) {
            context.session().logger().info("Address before create " + address);

            AddressStorage<AddressBuilder> storage = proxy.storage(address.ownerService());
            if (storage.storeNewValues()) {
                if (address.creationTime() == null) {
                    address.setCreationTime(getDateFieldOrNow(address.creationTime()));
                }

                if (address.modificationTime() == null) {
                    address.setModificationTime(getDateFieldOrNow(address.modificationTime()));
                }
                storage.create(context, address, new StorageSaveCallback(context));
                return;
            }

            Tuple stmt = Tuple.tuple();
            stmt.addString(context.userType());
            stmt.addString(String.valueOf(context.userId()));
            stmt.addString(address.id().addressId());
            stmt.addInteger(address.regionId());
            stmt.addInteger(address.preciseRegionId());
            stmt.addString(address.country());
            stmt.addString(address.city());
            stmt.addString(address.street());
            stmt.addString(address.building());
            stmt.addString(address.floor());
            stmt.addString(address.room());
            stmt.addString(address.entrance());
            stmt.addString(address.intercom());
            stmt.addString(address.zip());
            stmt.addString(address.subtype());
            stmt.addString(address.comment());
            stmt.addBigDecimal(address.location().getLatitude());
            stmt.addBigDecimal(address.location().getLongitude());
            stmt.addString(address.district());
            stmt.addString(address.platform());
            stmt.addString(address.ownerService().serviceName());
            stmt.addString(address.subtype());
            stmt.addBoolean(false);
            stmt.addBoolean(context.draft());
            OffsetDateTime creationTime = getDateFieldOrNow(address.creationTime());
            stmt.addLocalDateTime(creationTime.toLocalDateTime());
            OffsetDateTime modificationTime = getDateFieldOrNow(address.modificationTime());
            stmt.addLocalDateTime(modificationTime.toLocalDateTime());
            if (address.lastTouchedTime() != null) {
                stmt.addLocalDateTime(address.lastTouchedTime().toLocalDateTime());
            } else {
                stmt.addLocalDateTime(null);
            }
            stmt.addString(address.locale());

            address.setCreationTime(creationTime);
            address.setModificationTime(modificationTime);
            context.session().logger().info(stmt.toString());
            proxy.pgClient().executeOnMaster(
                INSERT_ADDRESS_VERTX,
                stmt,
                context.session().listener(),
                new InsertCallback(context, address));
        }
    }

    private class InsertCallback extends AbstractAddressSessionCallback<RowSet<Row>> {
        private final AddressContext context;
        private final PassportAddressBuilder address;

        public InsertCallback(
            final AddressContext context,
            final PassportAddressBuilder address) {
            super(context.session());
            this.context = context;
            this.address = address;
        }

        @Override
        public void completed(final RowSet<Row> rows) {
            try {
                if (rows.rowCount() <= 0) {
                    writeErrorResponse(context.session(), HttpStatus.SC_INTERNAL_SERVER_ERROR, "Update returned empty" +
                                                                                                   " response");
                    return;
                }

                Row resultSet = rows.iterator().next();
                Long id = resultSet.getLong("id");
                context.session().logger().info("Saved " + id);

                writeResults(address, context);

            } catch (Exception e) {
                context.session().logger().log(Level.WARNING, "Fail to create", e);
                try {
                    writeErrorResponse(
                        context.session(),
                        HttpStatus.SC_INTERNAL_SERVER_ERROR,
                        e.getMessage());
                } catch (IOException ioe) {
                    failed(ioe);
                }
            }
        }
    }

    private class StorageSaveCallback extends AbstractAddressSessionCallback<PassportAddressBuilder> {
        private final AddressContext context;

        public StorageSaveCallback(final AddressContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final PassportAddressBuilder address) {
            try {
                writeResults(address, context);
            } catch (HttpException | IOException e) {
                failed(e);
            }
        }
    }

    protected void writeResults(
        final PassportAddress address,
        final AddressContext context)
        throws HttpException, IOException
    {
        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        ProxySession session = context.session();
        JsonType type = JsonTypeExtractor.NORMAL.extract(session.params());

        boolean taxi = "taxi".equalsIgnoreCase(context.serviceName());

        try (OutputStreamWriter outWriter =
                 new OutputStreamWriter(
                     out,
                     CharsetUtils.acceptedCharset(session.request())
                         .newEncoder()
                         .onMalformedInput(
                             CodingErrorAction.REPLACE)
                         .onUnmappableCharacter(
                             CodingErrorAction.REPLACE));
             JsonWriter writer = type.create(outWriter))
        {
            writer.startObject();
            writer.key("status");
            writer.value("ok");
            if (taxi) {
                writer.key("id");
                writer.value(address.id());
                writer.key("version");
                writer.value(0);
            } else {
                writer.value(address);
            }
            writer.endObject();
        }

        NByteArrayEntity entity =
            out.processWith(NByteArrayEntityFactory.INSTANCE);
        entity.setContentType(
            ContentType.APPLICATION_JSON.withCharset(
                session.acceptedCharset())
                .toString());
        session.response(HttpStatus.SC_OK, entity);
    }

    private static OffsetDateTime getDateFieldOrNow(OffsetDateTime field) {
        return Objects.requireNonNullElseGet(field, OffsetDateTime::now);
    }
}
