package ru.yandex.passport.address.handlers;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.CodingErrorAction;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.passport.AbstractAddressSessionCallback;
import ru.yandex.passport.AddressContext;
import ru.yandex.passport.AddressProxy;
import ru.yandex.passport.AddressStorage;
import ru.yandex.passport.ReadWriteRights;
import ru.yandex.passport.address.AddressBuilder;
import ru.yandex.passport.address.AddressId;
import ru.yandex.passport.address.config.AddressServicesRights;
import ru.yandex.search.msal.pool.DBConnectionPool;

public class DeleteAddressHandler extends AbstractAddressHandler implements ProxyRequestHandler {
    private final DBConnectionPool passportConnectionPool;

    private static final String INSERT_HIDDEN_ADDRESS =
        "INSERT INTO passport_hidden_address(user_type, user_id, address_id, hidden_by_service, owner_service) " +
         "VALUES (?, ?, ?, ?, ?) RETURNING address_id";

    private final AddressProxy proxy;
    public DeleteAddressHandler(final AddressProxy proxy) {
        this.passportConnectionPool = proxy.pool(proxy.config().passportDbConfig());
        this.proxy = proxy;
    }

    @Override
    public void handleInternal(
        final ProxySession session)
        throws HttpException, IOException, SQLException, JsonException
    {
        //AddressId id = session.params().get("id", AddressIdParser.INSTANCE);
        AddressContext context = new AddressContext(session);
        AddressId id = parseWithTaxi(context);
        //PassportAddressUser user = PassportAddressUser.parse(session.params());
        ReadWriteRights rights = context.rights(id);
        if (rights == null || rights.ordinal() < ReadWriteRights.READ_WRITE.ordinal()) {
            if (rights == ReadWriteRights.READ_ONLY && AddressServicesRights.INSTANCE.hiddenEnabled(context.service())) {
                hideAddress(id, context);
            } else {
                writeErrorResponse(
                    session,
                    HttpStatus.SC_FORBIDDEN,
                    "Service " + context.serviceName()
                        + " does not have write rights for namespace " + id.ownerService());
            }
            return;
        }

        AddressStorage<AddressBuilder> storage = proxy.storage(id.ownerService());
        context.session().logger().info("Owner service: " + id.ownerService());
        storage.delete(context, id, new DeleteAddressCallback(context));
    }

    private static class DeleteAddressCallback extends AbstractAddressSessionCallback<Object> {
        private final AddressContext context;

        public DeleteAddressCallback(final AddressContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Object o) {
            ProxySession session = context.session();
            DecodableByteArrayOutputStream out =
                new DecodableByteArrayOutputStream();

            int status = HttpStatus.SC_OK;

            try (OutputStreamWriter outWriter =
                     new OutputStreamWriter(
                         out,
                         CharsetUtils.acceptedCharset(session.request())
                             .newEncoder()
                             .onMalformedInput(
                                 CodingErrorAction.REPLACE)
                             .onUnmappableCharacter(
                                 CodingErrorAction.REPLACE));
                 JsonWriter writer = JsonTypeExtractor.NORMAL.extract(session.params()).create(outWriter))
            {
                writer.startObject();
                writer.key("status");
                writer.value("ok");
                writer.endObject();
            } catch (IOException | HttpException e) {
                failed(e);
                return;
            }

            NByteArrayEntity entity =
                out.processWith(NByteArrayEntityFactory.INSTANCE);
            entity.setContentType(
                ContentType.APPLICATION_JSON.withCharset(
                        session.acceptedCharset())
                    .toString());
            session.response(status, entity);
        }
    }

    private void hideAddress(
        final AddressId addressId,
        final AddressContext context)
        throws HttpException, IOException, SQLException, JsonException
    {
        ProxySession session = context.session();
        try (Connection connection = passportConnectionPool.getConnection(session.logger());
             PreparedStatement stmt = connection.prepareStatement(INSERT_HIDDEN_ADDRESS))
        {
            //stmt.setString(1, user.userType());
            stmt.setString(1, context.userType());
            stmt.setLong(2, context.userIdLong());
            stmt.setString(3, addressId.toString());
            stmt.setString(4, context.serviceName());
            stmt.setString(5, addressId.ownerService().serviceName());
            session.logger().info(stmt.toString());
            try (ResultSet resultSet = stmt.executeQuery()) {
                writeResults(resultSet, session);
            }
        } catch (Exception e) {
            session.logger().log(Level.SEVERE, "Request failed", e);
            writeErrorResponse(session, HttpStatus.SC_INTERNAL_SERVER_ERROR, "Database request failure");
            return;
        }
    }

    protected void writeResults(
        final ResultSet resultSet,
        final ProxySession session)
        throws HttpException, IOException, SQLException
    {
        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        JsonType type = JsonTypeExtractor.NORMAL.extract(session.params());

        int status = HttpStatus.SC_OK;

        try (OutputStreamWriter outWriter =
                 new OutputStreamWriter(
                     out,
                     CharsetUtils.acceptedCharset(session.request())
                         .newEncoder()
                         .onMalformedInput(
                             CodingErrorAction.REPLACE)
                         .onUnmappableCharacter(
                             CodingErrorAction.REPLACE));
             JsonWriter writer = type.create(outWriter))
        {
            writer.startObject();
            boolean next = resultSet.next();
            if (next) {
                writer.key("status");
                writer.value("ok");
            } else {
                writer.key("status");
                writer.value("error");
                writer.key("error");
                writer.value("not found");
                status = HttpStatus.SC_NOT_FOUND;
            }
            writer.endObject();
        }

        NByteArrayEntity entity =
            out.processWith(NByteArrayEntityFactory.INSTANCE);
        entity.setContentType(
            ContentType.APPLICATION_JSON.withCharset(
                    session.acceptedCharset())
                .toString());
        session.response(status, entity);
    }
}
