package ru.yandex.passport.address.handlers;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.CodingErrorAction;
import java.sql.SQLException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.passport.AbstractAddressSessionCallback;
import ru.yandex.passport.AddressProxy;
import ru.yandex.passport.AddressStorage;
import ru.yandex.passport.address.AddressBuilder;
import ru.yandex.passport.address.AddressId;
import ru.yandex.passport.address.ListAddressContext;
import ru.yandex.passport.address.config.AddressServicesRights;

public class ListAddressHandler extends AbstractAddressHandler implements ProxyRequestHandler {
    private final AddressProxy proxy;

    public ListAddressHandler(final AddressProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void handleInternal(final ProxySession session) throws HttpException, IOException, SQLException, JsonException {
        ListAddressContext context = new ListAddressContext(session);

        DoubleFutureCallback<Set<AddressId>, List<List<AddressBuilder>>> hiddenAndAddresses
            = new DoubleFutureCallback<>(new PrintListCallback(context));

        if (AddressServicesRights.INSTANCE.hiddenEnabled(context.service())) {
            proxy.storage(context.service()).hidden(context, hiddenAndAddresses.first());
        } else {
            hiddenAndAddresses.first().completed(Collections.emptySet());
        }

        MultiFutureCallback<List<AddressBuilder>> mfcb = new MultiFutureCallback<>(hiddenAndAddresses.second());

        for (AddressStorage<AddressBuilder> storage: proxy.allowedReads(context.service())) {
            if (storage.optional()) {
                storage.list(
                    context,
                    new ErrorSuppressingFutureCallback<>(new NamedCallback(storage.name(), context.session(), mfcb.newCallback()), Collections.emptyList()));
            } else {
                storage.list(context, new NamedCallback(storage.name(), context.session(), mfcb.newCallback()));
            }

        }

        mfcb.done();
    }


    private static class NamedCallback extends FilterFutureCallback<List<AddressBuilder>> {
        private final String name;
        private final ProxySession session;

        public NamedCallback(final String name,
                             final ProxySession session,
                             final FutureCallback<? super List<AddressBuilder>> callback) {
            super(callback);
            this.name = name;
            this.session = session;
        }

        @Override
        public void completed(final List<AddressBuilder> result) {
            session.logger().info("NamedCompleted " + name);
            super.completed(result);
        }
    }

    private static class PrintListCallback
        extends AbstractAddressSessionCallback<Map.Entry<Set<AddressId>, List<List<AddressBuilder>>>>
    {
        private final ListAddressContext context;

        public PrintListCallback(final ListAddressContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Map.Entry<Set<AddressId>, List<List<AddressBuilder>>> hiddenAndAddresses) {
            DecodableByteArrayOutputStream out =
                new DecodableByteArrayOutputStream();
            JsonType type;
            try {
                type = JsonTypeExtractor.NORMAL.extract(context.session().params());
            } catch (BadRequestException bre) {
                failed(bre);
                return;
            }

            List<List<AddressBuilder>> addresses = hiddenAndAddresses.getValue();
            Set<AddressId> hidden = hiddenAndAddresses.getKey();
            context.session().logger().info("Hidden loaded " + hidden.size());

            try (OutputStreamWriter outWriter =
                     new OutputStreamWriter(
                         out,
                         CharsetUtils.acceptedCharset(context.session().request())
                             .newEncoder()
                             .onMalformedInput(
                                 CodingErrorAction.REPLACE)
                             .onUnmappableCharacter(
                                 CodingErrorAction.REPLACE));
                 JsonWriter writer = type.create(outWriter))
            {
                writer.startObject();
                writer.key("status");
                writer.value("ok");
                writer.key("addresses");
                writer.startArray();
                int cnt = 0;
                for (List<AddressBuilder> list: addresses) {
                    for (AddressBuilder address: list) {
                        cnt +=1;
                        if (hidden.contains(address.id())) {
                            continue;
                        }

                        if (!context.service().addressFilter().test(address)) {
                            context.session().logger().warning("Filtering out " + address);
                            continue;
                        }

                        writer.startObject();
                        writer.value(address.build());
                        writer.value(context.passportUser());
                        writer.endObject();

                        if (cnt >= context.length()) {
                            break;
                        }
                    }
                    if (cnt >= context.length()) {
                        break;
                    }
                }

                writer.endArray();
                writer.key("more");
                writer.value(cnt >= context.length());
                writer.endObject();
            } catch (IOException | HttpException ioe) {
                failed(ioe);
                return;
            }

            NByteArrayEntity entity =
                out.processWith(NByteArrayEntityFactory.INSTANCE);
            entity.setContentType(
                ContentType.APPLICATION_JSON.withCharset(
                        context.session().acceptedCharset())
                    .toString());
            context.session().response(HttpStatus.SC_OK, entity);
        }
    }
}
