package ru.yandex.passport.address.handlers;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.CodingErrorAction;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.passport.AbstractAddressSessionCallback;
import ru.yandex.passport.AddressContext;
import ru.yandex.passport.AddressProxy;
import ru.yandex.passport.AddressStorage;
import ru.yandex.passport.ReadWriteRights;
import ru.yandex.passport.address.Address;
import ru.yandex.passport.address.AddressBuilder;
import ru.yandex.passport.address.AddressId;
import ru.yandex.passport.address.AsyncGeocoderClient;
import ru.yandex.passport.address.PassportAddress;
import ru.yandex.passport.address.PassportAddressBuilder;
import ru.yandex.passport.address.PassportAddressDto;

public class UpdateAddressHandler extends AbstractAddressHandler implements ProxyRequestHandler {
    //private static final int WORKERS = 100;
    //private final DBConnectionPool passportConnectionPool;
    private final ThreadPoolExecutor executor;
    private final AddressProxy proxy;

    private static final String ADDRESSES = "addresses";
    private static final String ID = "id";
    private static final String USER_ID = "user_id";
    private static final String USER_TYPE = "user_type";
    private static final String UID = "uid";
    private static final String TAXI = "taxi";

    public UpdateAddressHandler(final AddressProxy proxy) {
        //this.passportConnectionPool = proxy.pool(proxy.config().passportDbConfig());
        this.proxy = proxy;
        this.executor =
            new ThreadPoolExecutor(
                proxy.config().passportDbConfig().poolSize(),
                proxy.config().passportDbConfig().poolSize(),
                1,
                TimeUnit.DAYS,
                new LinkedBlockingQueue<>());
    }

    @Override
    public void handleInternal(
        final ProxySession session)
        throws HttpException, IOException, SQLException, JsonException
    {
        if (!(session.request() instanceof HttpEntityEnclosingRequest)) {
            writeErrorResponse(session, HttpStatus.SC_BAD_REQUEST, "No body supplied for request");
            return;
        }

        JsonMap addressDto =
            TypesafeValueContentHandler.parse(
                CharsetUtils.content(
                    ((HttpEntityEnclosingRequest) session.request()).getEntity())).asMap();
        session.logger().info("AddrPrint;" + JsonType.NORMAL.toString(addressDto));
        JsonList addresses = addressDto.getListOrNull(ADDRESSES);

        final AddressContext context;
        if (addresses != null) {
            context = new AddressContext(session, false);
        } else {
            context = new AddressContext(session);
        }

        MultiFutureCallback<PassportAddressBuilder> mfcb = new MultiFutureCallback<>(new PrintCallback(context));

        boolean taxi = TAXI.equalsIgnoreCase(context.serviceName());

        if (addresses == null) {
            String idStr = session.params().get(ID, null, NonEmptyValidator.TRIMMED);
            if (idStr == null) {
                idStr = session.params().get(ID, NonEmptyValidator.TRIMMED);
            }
            PassportAddressBuilder address = parseAndCheck(context, addressDto, taxi);
            processAddress(context, address, idStr, mfcb.newCallback());
        } else {
            session.logger().info("Multiaddresses, size: " + addresses.size());
            Map<String, PassportAddressBuilder> addressBuilders = new HashMap<>();
            for (JsonObject addressObj: addresses) {
                JsonMap addressMap = addressObj.asMap();
                String idStr = addressMap.get(ID, NonEmptyValidator.TRIMMED);
                String userType = addressMap.getString(USER_TYPE, context.userType());
                Object userId;
                if (UID.equalsIgnoreCase(userType)) {
                    userId = addressMap.getLong(USER_ID);
                } else {
                    userId = addressMap.getString(USER_ID);
                }
                if (userId == null) {
                    if (context.userId() != null) {
                        userId = context.userId();
                    } else {
                        throw new BadRequestException("No user id parameter set");
                    }
                }

                final AddressContext singleContext = new AddressContext(context, userType, userId);
                final PassportAddressBuilder address =
                    parseAndCheck(singleContext, addressMap, taxi);
                final FutureCallback<PassportAddressBuilder> callback = mfcb.newCallback();
                executor.execute(() -> {
                        try {
                            processAddress(singleContext, address, idStr, callback);
                        } catch (Exception e) {
                            callback.failed(e);
                        }
                });

                addressBuilders.put(idStr, address);
            }
        }
        mfcb.done();
    }

    private PassportAddressBuilder parseAndCheck(
        final AddressContext context,
        final JsonMap map,
        final boolean taxi)
        throws JsonException, BadRequestException
    {
        PassportAddressBuilder address = PassportAddressDto.parseForCreate(context, map);
        if (!taxi && !address.validForGeocoder()) {
            throw new BadRequestException(
                "address_line or country,city,street or district,building should be specified");
        }
        return address;
    }

    protected void processAddress(
        final AddressContext context,
        final PassportAddressBuilder address,
        final String idStr,
        final FutureCallback<PassportAddressBuilder> callback)
        throws HttpException, IOException
    {
        ProxySession session = context.session();

        AddressId id = parseWithTaxi(context, idStr);

        //PassportAddressUser user = PassportAddressUser.parse(session.params());
        if (context.rights(id).ordinal() < ReadWriteRights.READ_WRITE.ordinal()) {
            writeErrorResponse(
                session,
                HttpStatus.SC_FORBIDDEN,
                "Service " + context.serviceName()
                    + " does not have write rights for namespace " + id.ownerService());
            return;
        }

        address.setId(id);
        proxy.storage(id.ownerService()).deletePassportTranslatedAddress(session, id);

        FillCallback fillCallback = new FillCallback(context, callback);

        if (!context.serviceConfig().useGeocoderOnUpdate()) {
            fillCallback.completed(address);
        } else {
            AsyncGeocoderClient geocoderClient = proxy.geocoderClient().adjust(session);
            geocoderClient.fillAddress(address, fillCallback);
        }
    }

    private static class PrintCallback extends AbstractAddressSessionCallback<List<PassportAddressBuilder>> {
        private final AddressContext context;
        public PrintCallback(final AddressContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final List<PassportAddressBuilder> addresses) {
            try {
                DecodableByteArrayOutputStream out =
                    new DecodableByteArrayOutputStream();
                ProxySession session = context.session();
                JsonType type = JsonTypeExtractor.NORMAL.extract(session.params());
                try (OutputStreamWriter outWriter =
                         new OutputStreamWriter(
                             out,
                             CharsetUtils.acceptedCharset(session.request())
                                 .newEncoder()
                                 .onMalformedInput(
                                     CodingErrorAction.REPLACE)
                                 .onUnmappableCharacter(
                                     CodingErrorAction.REPLACE));
                     JsonWriter writer = type.create(outWriter))
                {
                    writer.startObject();
                    writer.key("status");
                    writer.value("ok");
                    if (context.returnUpdated()) {
                        if (context.taxi() || addresses.size() > 1) {
                            writer.key(ADDRESSES);
                            writer.startArray();
                            for (Address address: addresses) {
                                writer.startObject();
                                writer.value(address);
                                writer.endObject();
                            }
                            writer.endArray();
                        } else {
                            writer.value(addresses.get(0));
                        }
                    }
//                    if (context.returnUpdated()) {
//                        writer.value(address);
//                    }
                    writer.endObject();
                }

                NByteArrayEntity entity =
                    out.processWith(NByteArrayEntityFactory.INSTANCE);
                entity.setContentType(
                    ContentType.APPLICATION_JSON.withCharset(
                            session.acceptedCharset())
                        .toString());
                session.response(HttpStatus.SC_OK, entity);
            } catch (Exception e) {
                failed(e);
            }

        }
    }

    private class FillCallback extends AbstractProxySessionCallback<PassportAddressBuilder> {
        private final AddressContext context;
        private final FutureCallback<PassportAddressBuilder> callback;

        public FillCallback(final AddressContext context, final FutureCallback<PassportAddressBuilder> callback) {
            super(context.session());
            this.context = context;
            this.callback = callback;
        }

        @Override
        public void completed(final PassportAddressBuilder address) {
            context.session().logger().info("Address before update " + address);

            AddressStorage<AddressBuilder> storage = proxy.storage(address.ownerService());
            storage.update(context, address, callback);
        }
    }

//    private static class UpdateCallback extends AbstractAddressSessionCallback<PassportAddressBuilder> {
//        private final AddressContext context;
//
//        public UpdateCallback(final AddressContext context) {
//            super(context.session());
//            this.context = context;
//        }
//
//        @Override
//        public void completed(final PassportAddressBuilder addressBuilder) {
//        }
//    }

    protected void writeResults(
        final PassportAddress address,
        final AddressContext context)
        throws HttpException, IOException, SQLException
    {
        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        ProxySession session = context.session();
        JsonType type = JsonTypeExtractor.NORMAL.extract(session.params());
        int status = HttpStatus.SC_OK;
        try (OutputStreamWriter outWriter =
                 new OutputStreamWriter(
                     out,
                     CharsetUtils.acceptedCharset(session.request())
                         .newEncoder()
                         .onMalformedInput(
                             CodingErrorAction.REPLACE)
                         .onUnmappableCharacter(
                             CodingErrorAction.REPLACE));
             JsonWriter writer = type.create(outWriter))
        {
            writer.startObject();
            writer.key("status");
            writer.value("ok");
            if (context.returnUpdated()) {
                writer.value(address);
            }
            writer.endObject();
        }

        NByteArrayEntity entity =
            out.processWith(NByteArrayEntityFactory.INSTANCE);
        entity.setContentType(
            ContentType.APPLICATION_JSON.withCharset(
                    session.acceptedCharset())
                .toString());
        session.response(status, entity);
    }
}
