package ru.yandex.passport.contact;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.OffsetDateTime;
import java.util.Map;

import ru.yandex.blackbox.BlackboxAddress;
import ru.yandex.blackbox.BlackboxDbfield;
import ru.yandex.blackbox.BlackboxPhoneAttributeType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.passport.AddressServiceParser;
import ru.yandex.passport.address.AddressId;
import ru.yandex.passport.address.AddressService;

public class ContactBuilder implements Contact {
    private AddressId id;
    private String firstName;
    private String lastName;
    private String secondName;
    private String phoneNum;
    private String email;
    private Object userId;
    private String userType;
    private AddressService ownerService;
    private OffsetDateTime lastTouchedTime;
    private OffsetDateTime creationTime;
    private OffsetDateTime modificationTime;

    public ContactBuilder() {
    }

    @Override
    public AddressService ownerService() {
        return ownerService;
    }

    public void ownerService(final AddressService ownerService) {
        this.ownerService = ownerService;
    }

    @Override
    public AddressId id() {
        return id;
    }

    public ContactBuilder id(final AddressId id) {
        this.id = id;
        return this;
    }

    @Override
    public String firstName() {
        return firstName;
    }

    public ContactBuilder firstName(final String firstName) {
        this.firstName = firstName;
        return this;
    }

    @Override
    public String lastName() {
        return lastName;
    }

    public ContactBuilder lastName(final String lastName) {
        this.lastName = lastName;
        return this;
    }

    @Override
    public String secondName() {
        return secondName;
    }

    public ContactBuilder secondName(final String secondName) {
        this.secondName = secondName;
        return this;
    }

    @Override
    public String phoneNum() {
        return phoneNum;
    }

    public ContactBuilder phoneNum(final String phoneNum) {
        this.phoneNum = phoneNum;
        return this;
    }

    @Override
    public String email() {
        return email;
    }

    public ContactBuilder email(final String email) {
        this.email = email;
        return this;
    }

    @Override
    public Object userId() {
        return userId;
    }

    public ContactBuilder userId(final Object userId) {
        this.userId = userId;
        return this;
    }

    @Override
    public String userType() {
        return userType;
    }

    public ContactBuilder userType(final String userType) {
        this.userType = userType;
        return this;
    }

    @Override
    public OffsetDateTime lastTouchedTime() {
        return lastTouchedTime;
    }

    public ContactBuilder lastTouchedTime(final OffsetDateTime lastTouchedTime) {
        this.lastTouchedTime = lastTouchedTime;
        return this;
    }

    @Override
    public OffsetDateTime creationTime() {
        return creationTime;
    }

    public ContactBuilder creationTime(final OffsetDateTime creationTime) {
        this.creationTime = creationTime;
        return this;
    }

    @Override
    public OffsetDateTime modificationTime() {
        return modificationTime;
    }

    public ContactBuilder modificationTime(final OffsetDateTime modificationTime) {
        this.modificationTime = modificationTime;
        return this;
    }

    public static ContactBuilder build(
        final Object userId,
        final String userType,
        final ResultSet result)
        throws SQLException
    {
        ContactBuilder builder = new ContactBuilder();
        String ownerServiceName = result.getString("owner_service");
        builder.ownerService(AddressServiceParser.INSTANCE.apply(ownerServiceName));
        builder.id(new AddressId(userType, userId, builder.ownerService(), result.getString("object_key")));
        builder.lastName(result.getString("last_name"));
        builder.firstName(result.getString("first_name"));
        builder.secondName(result.getString("second_name"));
        builder.email(result.getString("email"));
        builder.phoneNum(result.getString("phone_number"));

        builder.lastTouchedTime(result.getObject("last_touched_time", OffsetDateTime.class));
        builder.creationTime(result.getObject("creation_time", OffsetDateTime.class));
        builder.modificationTime(result.getObject("modification_time", OffsetDateTime.class));
        builder.userType(userType);
        builder.userId(userId);

        return builder;
    }

    public static ContactBuilder fromBlackboxUserInfo(
        final PrefixedLogger logger,
        final Object userId,
        final String userType,
        final BlackboxUserinfo userinfo,
        final String id)
    {
        ContactBuilder builder = new ContactBuilder();
        builder.ownerService(AddressService.PASSPORT);
        builder.id(new AddressId(userType, userId, builder.ownerService(), "blackbox-" + id));
        builder.userId(userId);
        builder.userId(userType);
        if (userinfo.addressList().size() > 0) {
            BlackboxAddress bbEmail = userinfo.addressList().get(0);
            for (BlackboxAddress address: userinfo.addressList()) {
                if (address.defaultFlag()) {
                    bbEmail = address;
                }
            }
            builder.email(bbEmail.email());
        } else {
            return null;
        }
        String bbPhone = null;
        for (Map.Entry<String, Map<BlackboxPhoneAttributeType, String>> entry:  userinfo.phoneAttributes().entrySet()) {
            bbPhone = entry.getValue().get(BlackboxPhoneAttributeType.PHONE_NUMBER);
            if (bbPhone != null && !bbPhone.isEmpty()) {
                break;
            }
        }
        if (bbPhone == null) {
            return null;
        }
        builder.phoneNum(ContactUtils.normalizePhoneNumber(logger, bbPhone));
        builder.firstName(userinfo.dbfields().get(BlackboxDbfield.FIRSTNAME));
        builder.lastName(userinfo.dbfields().get(BlackboxDbfield.LASTNAME));

        return builder;
    }

    public static Integer parseBlackboxId(final String id) {
        int index = id.indexOf('-');
        if (index >= id.length() - 1) {
            return null;
        }
        try {
            return Integer.parseInt(id.substring(index + 1));
        } catch (NumberFormatException nfe) {
            return null;
        }
    }
}
