package ru.yandex.passport.contact.handlers;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.passport.AddressContext;
import ru.yandex.passport.AddressProxy;
import ru.yandex.passport.ReadWriteRights;
import ru.yandex.passport.address.AddressId;
import ru.yandex.passport.address.AddressIdParser;
import ru.yandex.passport.address.config.AddressServicesRights;
import ru.yandex.search.msal.pool.DBConnectionPool;

public class DeleteContactHandler extends AbstractContactHandler {
    private final DBConnectionPool passportConnectionPool;
    private static final String DELETE_PASSPORT_CONTACT
        = "UPDATE passport_contact as c SET deleted = true WHERE c.object_key = ? RETURNING id";
    private static final String INSERT_HIDDEN_CONTACT =
        "INSERT INTO passport_hidden_contact(user_type, user_id, contact_id, hidden_by_service, owner_service) " +
            "VALUES (?, ?, ?, ?, ?) RETURNING contact_id";

    public DeleteContactHandler(final AddressProxy proxy) {
        this.passportConnectionPool = proxy.pool(proxy.config().passportDbConfig());
    }

    @Override
    public void handleInternal(
        final ProxySession session)
        throws HttpException, IOException, SQLException, JsonException
    {
        AddressId id = session.params().get("id", AddressIdParser.INSTANCE);
        AddressContext context = new AddressContext(session);
        ReadWriteRights rights = context.rights(id);
        if (rights == null || rights.ordinal() < ReadWriteRights.READ_WRITE.ordinal()) {
            if (rights == ReadWriteRights.READ_ONLY && AddressServicesRights.INSTANCE.hiddenEnabled(context.service())) {
                hideContact(id, context);
            } else {
                writeErrorResponse(
                    session,
                    HttpStatus.SC_FORBIDDEN,
                    "Service " + context.serviceName()
                        + " does not have write rights for namespace " + id.ownerService());
            }

            return;
        }

        try (Connection connection = passportConnectionPool.getConnection(session.logger());
             PreparedStatement stmt = connection.prepareStatement(DELETE_PASSPORT_CONTACT))
        {
            stmt.setString(1, id.addressId());
            session.logger().info(stmt.toString());
            try (ResultSet resultSet = stmt.executeQuery()) {
                writeResults(resultSet, context);
            }
        } catch (Exception e) {
            session.logger().log(Level.SEVERE, "Request failed", e);
            writeErrorResponse(session, HttpStatus.SC_INTERNAL_SERVER_ERROR, "Database request failure");
            return;
        }
    }

    private void hideContact(
        final AddressId addressId,
        final AddressContext context)
        throws HttpException, IOException, SQLException, JsonException
    {
        ProxySession session = context.session();
        try (Connection connection = passportConnectionPool.getConnection(session.logger());
             PreparedStatement stmt = connection.prepareStatement(INSERT_HIDDEN_CONTACT))
        {
            //stmt.setString(1, user.userType());
            stmt.setString(1, context.userType());
            stmt.setLong(2, context.userIdLong());
            stmt.setString(3, addressId.toString());
            stmt.setString(4, context.serviceName());
            stmt.setString(5, addressId.ownerService().serviceName());
            session.logger().info(stmt.toString());
            try (ResultSet resultSet = stmt.executeQuery()) {
                writeResults(resultSet, context);
            }
        } catch (Exception e) {
            session.logger().log(Level.SEVERE, "Request failed", e);
            writeErrorResponse(session, HttpStatus.SC_INTERNAL_SERVER_ERROR, "Database request failure");
            return;
        }
    }

    protected void writeResults(
        final ResultSet resultSet,
        final AddressContext context)
        throws HttpException, IOException, SQLException
    {
        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();

        int status = HttpStatus.SC_OK;

        try (OutputStreamWriter outWriter = outWriter(context.session(), out);
             JsonWriter writer = context.jsonType().create(outWriter))
        {
            writer.startObject();
            boolean next = resultSet.next();
            if (next) {
                writer.key("status");
                writer.value("ok");
            } else {
                writer.key("status");
                writer.value("error");
                writer.key("error");
                writer.value("not found");
                status = HttpStatus.SC_NOT_FOUND;
            }
            writer.endObject();
        }

        NByteArrayEntity entity =
            out.processWith(NByteArrayEntityFactory.INSTANCE);
        entity.setContentType(
            ContentType.APPLICATION_JSON.withCharset(
                    context.session().acceptedCharset())
                .toString());
        context.session().response(status, entity);
    }
}
