package ru.yandex.passport.contact.handlers;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxEmailsType;
import ru.yandex.blackbox.BlackboxNotFoundException;
import ru.yandex.blackbox.BlackboxPhoneAttributeType;
import ru.yandex.blackbox.BlackboxUserIdType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.blackbox.BlackboxUserinfos;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.passport.AbstractAddressSessionCallback;
import ru.yandex.passport.AddressContext;
import ru.yandex.passport.AddressProxy;
import ru.yandex.passport.address.AddressId;
import ru.yandex.passport.address.AddressService;
import ru.yandex.passport.address.config.AddressServicesRights;
import ru.yandex.passport.contact.Contact;
import ru.yandex.passport.contact.ContactBuilder;
import ru.yandex.search.msal.pool.DBConnectionPool;

public class ListContactHandler extends AbstractContactHandler {
    private static final String PASSPORT_LIST_CONTACTS =
        "SELECT " + PSSPORT_SELECT_COLUMNS_STMT +
            " FROM passport_contact as c " +
            "WHERE c.user_id=? AND c.user_type=? AND c.owner_service" +
            " =? AND NOT c.deleted = true LIMIT ?";
    private static final String LOAD_HIDDEN_CONTACTS =
        "SELECT contact_id FROM passport_hidden_contact " +
            "WHERE user_id=? AND user_type=? AND hidden_by_service=? LIMIT 300";

    private final DBConnectionPool passportConnectionPool;
    private final AddressProxy proxy;

    public ListContactHandler(final AddressProxy proxy) {
        this.passportConnectionPool = proxy.pool(proxy.config().passportDbConfig());
        this.proxy = proxy;
    }

    @Override
    public void handleInternal(final ProxySession session)
        throws HttpException, IOException, SQLException,JsonException
    {
        AddressContext context = new AddressContext(session);

        DoubleFutureCallback<Set<AddressId>, ContactList> hiddenAndContacts
            = new DoubleFutureCallback<>(new ListPrintCallback(context));

        if (AddressServicesRights.INSTANCE.hiddenEnabled(context.service())) {
            loadHidden(context, hiddenAndContacts.first());
        } else {
            hiddenAndContacts.first().completed(Collections.emptySet());
        }

        BlackboxClient blackboxClient = proxy.blackboxClient().adjust(session.context());
        BlackboxUserinfoRequest request =
            new BlackboxUserinfoRequest(BlackboxUserIdType.UID, context.userId())
                .dbfields(BLACKBOX_FIELDS)
                .emailsType(BlackboxEmailsType.GETDEFAULT)
                .phoneAttributes(BlackboxPhoneAttributeType.PHONE_NUMBER);

        blackboxClient.userinfo(request, session.listener()
                .createContextGeneratorFor(blackboxClient),
            new BlackboxCallback(context, hiddenAndContacts.second()));
    }

    public void loadHidden(final AddressContext context, final FutureCallback<Set<AddressId>> callback) {
        ProxySession session = context.session();
        try (Connection connection = passportConnectionPool.getConnection(session.logger());
             PreparedStatement stmt = connection.prepareStatement(LOAD_HIDDEN_CONTACTS))
        {

            session.logger().info(stmt.toString());
            //stmt.setString(1, user.userType());
            stmt.setString(1, String.valueOf(context.userId()));
            stmt.setString(2, context.userType());
            stmt.setString(3, context.serviceName());

            Set<AddressId> result = new LinkedHashSet<>();
            try (ResultSet resultSet = stmt.executeQuery()) {
                while (resultSet.next()) {
                    result.add(AddressId.parseOuterId(resultSet.getString("contact_id")));
                }
            }

            callback.completed(result);
        } catch (SQLException | IOException e) {
            callback.failed(e);
            return;
        }
    }

    private class ListPrintCallback extends AbstractAddressSessionCallback<Map.Entry<Set<AddressId>, ContactList>> {
        private final AddressContext context;

        public ListPrintCallback(final AddressContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Map.Entry<Set<AddressId>, ContactList> result) {
            DecodableByteArrayOutputStream out =
                new DecodableByteArrayOutputStream();

            ContactList contactList = result.getValue();
            Set<AddressId> hidden = result.getKey();
            try (OutputStreamWriter outWriter = outWriter(session(), out);
                 JsonWriter writer = context.jsonType().create(outWriter))
            {
                writer.startObject();
                writer.key("status");
                writer.value("ok");
                writer.key("contacts");
                writer.startArray();
                for (Contact contact : contactList.contacts()) {
                    if (hidden.contains(contact.id())) {
                        continue;
                    }
                    writer.startObject();
                    writer.value(contact);
                    writer.endObject();
                }
                writer.endArray();
                writer.key("more");
                writer.value(contactList.more());
                writer.endObject();
            } catch (IOException | HttpException ioe) {
                failed(ioe);
                return;
            }

            NByteArrayEntity entity =
                out.processWith(NByteArrayEntityFactory.INSTANCE);
            entity.setContentType(
                ContentType.APPLICATION_JSON.withCharset(
                        session().acceptedCharset())
                    .toString());
            session().response(HttpStatus.SC_OK, entity);
        }
    }

    private void handleWithBlackboxResolved(
        final AddressContext context,
        final FutureCallback<? super ContactList> callback,
        final BlackboxUserinfos userinfos)
    {
        final ProxySession session = context.session();
        int length;
        try {
            length = session.params().get("length", 20, PositiveIntegerValidator.INSTANCE);
        } catch (HttpException e) {
            callback.failed(e);
            return;
        }

        List<Contact> contacts = new ArrayList<>(length);
        int i = 0;
        for (BlackboxUserinfo userinfo : userinfos) {
            session.logger().info("Blackbox emails: " + userinfo.addressList());
            i += 1;
            Contact contact = ContactBuilder.fromBlackboxUserInfo(
                context.session().logger(),
                context.userId(),
                context.userType(),
                userinfo,
                String.valueOf(i));
            if (contact == null) {
                session.logger().warning("Null contact from bb info " + userinfo);
                continue;
            }
            contacts.add(contact);
        }
        int found = contacts.size();

        for (AddressService service : context.allowReads()) {
            if (found >= length) {
                break;
            }

            try (Connection connection = passportConnectionPool.getConnection(session.logger());
                 PreparedStatement stmt = connection.prepareStatement(PASSPORT_LIST_CONTACTS)) {
                //session.logger().info(PASSPORT_LIST_CONTACTS);
                //stmt.setString(1, user.userType());
                stmt.setString(1, String.valueOf(context.userId()));
                stmt.setString(2, context.userType());
                stmt.setString(3, service.serviceName());
                stmt.setInt(4, length);

                session.logger().info(stmt.toString());
                try (ResultSet resultSet = stmt.executeQuery()) {
                    while (resultSet.next()) {
                        contacts.add(
                            ContactBuilder.build(context.userId(), context.userType(), resultSet));
                        found += 1;
                    }
                }
            } catch (SQLException e) {
                callback.failed(e);
                return;
            }
        }

        callback.completed(new ContactList(contacts, found >= length));
    }

    private class BlackboxCallback extends AbstractFilterFutureCallback<BlackboxUserinfos, ContactList> {
        private final AddressContext context;

        public BlackboxCallback(final AddressContext context, final FutureCallback<? super ContactList> callback) {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final BlackboxUserinfos infos) {
            handleWithBlackboxResolved(context, callback, infos);
        }

        @Override
        public void failed(Exception e) {
            if (e instanceof BlackboxNotFoundException) {
                handleWithBlackboxResolved(context, callback, new BlackboxUserinfos(0));
                return;
            }
            super.failed(e);
        }
    }

    private static class ContactList {
        private final List<Contact> contacts;
        private final boolean more;

        public ContactList(final List<Contact> contacts, final boolean more) {
            this.contacts = contacts;
            this.more = more;
        }

        public List<Contact> contacts() {
            return contacts;
        }

        public boolean more() {
            return more;
        }
    }
}
