package ru.yandex.passport.storage;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.time.OffsetDateTime;
import java.util.Collections;
import java.util.List;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.common.util.geocoder.GeoSearchParams;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonDouble;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.passport.AddressContext;
import ru.yandex.passport.AddressProxy;
import ru.yandex.passport.AddressStorage;
import ru.yandex.passport.address.AddressBuilder;
import ru.yandex.passport.address.AddressId;
import ru.yandex.passport.address.AddressService;
import ru.yandex.passport.address.AsyncGeocoderClient;
import ru.yandex.passport.address.ListAddressContext;
import ru.yandex.passport.address.Location;
import ru.yandex.passport.address.PassportAddressBuilder;

public class HomeWorkDataSyncStorage implements AddressStorage<AddressBuilder> {
    private static final AddressService OWNER_SERVICE = AddressService.MAPS;

    private final HttpHost host;
    private final AsyncClient dataSyncClient;
    private final AsyncGeocoderClient geocoderClient;

    public HomeWorkDataSyncStorage(final AddressProxy proxy) {
        this.dataSyncClient = proxy.homeWorkDatasyncClient();
        this.host = proxy.config().homeWorkDatasyncConfig().host();
        this.geocoderClient = proxy.geocoderClient();
    }

    @Override
    public void list(final ListAddressContext context, final FutureCallback<List<AddressBuilder>> callback) {
        dataSyncClient.execute(
            host,
            new BasicAsyncRequestProducerGenerator(
                "/v2/" + context.userId() + "/personality/profile/addresses"),
            JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING_OK,
            context.session().listener().createContextGeneratorFor(dataSyncClient),
            new DataSyncListCallback(callback, context));
    }

    @Override
    public void get(final AddressContext context, final AddressId id, final FutureCallback<AddressBuilder> callback) {
        dataSyncClient.execute(
            host,
            new BasicAsyncRequestProducerGenerator(
                "/v2/" + context.userId() + "/personality/profile/addresses/" + id.addressId()),
            JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING_OK,
            context.session().listener().createContextGeneratorFor(dataSyncClient),
            new DataSyncGetCallback(callback, context));
    }

    @Override
    public void create(
        final AddressContext context,
        final PassportAddressBuilder address,
        final FutureCallback<PassportAddressBuilder> callback)
    {
        update(context, address, callback);
    }

    @Override
    public void update(
        final AddressContext context,
        final PassportAddressBuilder address,
        final FutureCallback<PassportAddressBuilder> callback)
    {
        boolean home = "home".equalsIgnoreCase(address.label());
        boolean work = "work".equalsIgnoreCase(address.label());
        if (!home && !work) {
            callback.failed(new BadRequestException("Label for " + address.ownerService() + " should be home or work, but got " + address.label()));
            return;
        }
        if (home) {
            address.setId(new AddressId(context.userType(), context.userId(), address.ownerService(), "home"));
        }

        if (work) {
            address.setId(new AddressId(context.userType(), context.userId(), address.ownerService(), "work"));
        }

        String serialized = JsonType.NORMAL.toString(toDataSync(address));
        context.session().logger().info("Saving homework " + serialized);
        try {
            dataSyncClient.execute(
                host,
                new BasicAsyncRequestProducerGenerator(
                    "/v2/" + context.userId() + "/personality/profile/addresses/" + address.id().addressId(),
                    new NByteArrayEntityGenerator(
                    new NStringEntity(
                        serialized,
                        ContentType.APPLICATION_JSON.withCharset(StandardCharsets.UTF_8))),
                    "PUT"),
                JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING_OK,
                context.session().listener().createContextGeneratorFor(dataSyncClient),
                new DataSyncCreateCallback(callback, address, context));
        } catch (IOException e) {
            callback.failed(e);
        }
    }

    @Override
    public void delete(
        final AddressContext context,
        final AddressId id,
        final FutureCallback<Object> callback)
    {
        BasicAsyncRequestProducerGenerator generator = new BasicAsyncRequestProducerGenerator(
            "/v2/" + context.userId() + "/personality/profile/addresses/" + id.addressId(),
            null,
            "DELETE");

        dataSyncClient.execute(
            host,
            generator,
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.ANY_GOOD,
                JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING),
            context.session().listener().createContextGeneratorFor(dataSyncClient),
            callback);
    }

    private class DataSyncGetCallback extends AbstractFilterFutureCallback<JsonObject, AddressBuilder> {
        private final AddressContext context;

        public DataSyncGetCallback(
            final FutureCallback<? super AddressBuilder> callback,
            final AddressContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObject) {
            try {
                JsonMap map = resultObject.asMap();
                PassportAddressBuilder addressBuilder =
                    parseFromDataSync(map, context.userType(), context.userId());
                geocoderClient.fillAddress(addressBuilder, callback);
            } catch (Exception e) {
                failed(e);
            }
        }
    }

    private static class DataSyncCreateCallback extends AbstractFilterFutureCallback<JsonObject, PassportAddressBuilder> {
        private final AddressContext context;
        private final PassportAddressBuilder address;

        public DataSyncCreateCallback(
            final FutureCallback<? super PassportAddressBuilder> callback,
            final PassportAddressBuilder address,
            final AddressContext context)
        {
            super(callback);
            this.context = context;
            this.address = address;
        }

        @Override
        public void completed(final JsonObject resultObject) {
            try {
                context.session().logger().info("SAve returned " + JsonType.NORMAL.toString(resultObject));
                callback.completed(address);
//                JsonMap map = resultObject.asMap();
//                PassportAddressBuilder addressBuilder =
//                    parseFromDataSync(map, context.userType(), context.userId());
//                geocoderClient.fillAddress(addressBuilder, callback);
            } catch (Exception e) {
                failed(e);
            }
        }
    }

    private class DataSyncListCallback extends AbstractFilterFutureCallback<JsonObject, List<AddressBuilder>> {
        private final AddressContext context;

        public DataSyncListCallback(
            final FutureCallback<? super List<AddressBuilder>> callback,
            final AddressContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObject) {
            try {
                context.session().logger().info("DataSync homework completed");
                JsonList items = resultObject.asMap().getList("items");
                if (items.size() == 0) {
                    callback.completed(Collections.emptyList());
                    return;
                }

                MultiFutureCallback<AddressBuilder> mfcb = new MultiFutureCallback<>(callback);
                for (JsonObject itemObj: items) {
                    JsonMap map = itemObj.asMap();
                    PassportAddressBuilder addressBuilder =
                        parseFromDataSync(map, context.userType(), context.userId());
                    GeoSearchParams.Builder paramBuilder = GeoSearchParams.builder();

                    Location location = addressBuilder.location();
                    if (location != null && location.getLongitude() != null && location.getLatitude() != null) {
                        paramBuilder = paramBuilder.withSearchAreaCenter(
                            location.getLongitude().doubleValue(),
                            location.getLatitude().doubleValue()
                        );
                    }

                    GeoSearchParams params = paramBuilder.build();
                    geocoderClient.fillAddress(addressBuilder, params, mfcb.newCallback());
                }

                mfcb.done();
            } catch (Exception e) {
                failed(e);
            }
        }
    }

    public static PassportAddressBuilder parseFromDataSync(
        final JsonMap map,
        final String userType,
        final Object userId)
        throws JsonException
    {
        PassportAddressBuilder builder = new PassportAddressBuilder();
        builder.setMarketAddressLine(map.getString("address_line"));
        builder.setType(map.getString("address_id"));
        AddressId addressId = new AddressId(userType, userId, OWNER_SERVICE, map.getString("address_id"));
        builder.setId(addressId);
        builder.setOwnerService(addressId.ownerService());
        builder.setLabel(map.getString("title", null));

        double latitude = map.getDouble("latitude");
        double longitude = map.getDouble("longitude");
        builder.setLocation(
            new Location(
                new BigDecimal(latitude),
                new BigDecimal(longitude))
        );

        try {
            String mtime = map.getString("modified", null);
            if (mtime != null) {
                builder.setModificationTime(OffsetDateTime.parse(mtime));
            }

            String ctime = map.getString("created", null);
            if (ctime != null) {
                builder.setCreationTime(OffsetDateTime.parse(ctime));
            }

            String utime = map.getString("last_used", null);
            if (utime != null) {
                builder.setLastTouchedTime(OffsetDateTime.parse(utime));
            }
        } catch (Exception e) {
            throw new JsonException(e);
        }

        return builder;
    }

    public static JsonMap toDataSync(final PassportAddressBuilder address) {
        JsonMap map = new JsonMap(BasicContainerFactory.INSTANCE);
        if ("work".equalsIgnoreCase(address.label())) {
            map.put("title", new JsonString("Work"));
        } else if ("home".equalsIgnoreCase(address.label())) {
            map.put("title", new JsonString("Home"));
        } else {
            return null;
        }

        map.put("address_id", new JsonString(address.id().addressId()));
        map.put("address_line", new JsonString(address.addressLine()));
        map.put("latitude", new JsonDouble(address.location().getLatitude().doubleValue()));
        map.put("longitude", new JsonDouble(address.location().getLongitude().doubleValue()));
        //map.put("address_line_short", new JsonString())
        if (address.entrance() != null) {
            map.put("entrance_number", new JsonString(address.entrance()));
        }
        map.put("created", new JsonString(address.creationTime().toString()));
        map.put("modified", new JsonString(address.modificationTime().toString()));

        return map;
    }

    @Override
    public boolean storeNewValues() {
        return true;
    }

    @Override
    public boolean optional() {
        return true;
    }
}
