package ru.yandex.passport.storage;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.ServerException;
import ru.yandex.passport.AddressContext;
import ru.yandex.passport.AddressProxy;
import ru.yandex.passport.AddressStorage;
import ru.yandex.passport.address.AddressBuilder;
import ru.yandex.passport.address.AddressId;
import ru.yandex.passport.address.ListAddressContext;
import ru.yandex.passport.address.MarketAddressBuilder;
import ru.yandex.passport.address.PassportAddressBuilder;
import ru.yandex.passport.address.handlers.AbstractAddressHandler;
import ru.yandex.search.msal.pool.DBConnectionPool;

public class MarketPgStorage implements AddressStorage<AddressBuilder> {
    //private static final String OWNER_SERVICE = "market";
    private static final String MARKET_LIST_ADDRESSES =
        "SELECT DISTINCT ON (a.region_id, a.country, a.city, a.street, a.building, a.floor, a.room, a.entrance, a.intercom, a.zip) " +
            AbstractAddressHandler.MARKET_COLUMNS_STMT + " FROM address as a " +
            "WHERE a.user_id=? AND a.user_type=? AND NOT a.deleted LIMIT ?";

    private static final String GET_MARKET_ADDRESS =
        "SELECT " + AbstractAddressHandler.MARKET_COLUMNS_STMT +
            " FROM address as a " +
            "WHERE a.object_key=?";
    private static final String DELETE_MARKET_ADDRESS =
        "UPDATE address as a SET deleted = true  " +
            "WHERE a.object_key=? RETURNING id";

    private final DBConnectionPool marketConnectionPool;

    public MarketPgStorage(final AddressProxy proxy) {
        this.marketConnectionPool = proxy.pool(proxy.config().marketDbConfig());
    }

    @Override
    public void list(
        final ListAddressContext context,
        final FutureCallback<List<AddressBuilder>> callback) {
        ProxySession session = context.session();
        try (Connection connection = marketConnectionPool.getConnection(session.logger());
             PreparedStatement stmt = connection.prepareStatement(MARKET_LIST_ADDRESSES)) {
            //stmt.setString(1, user.userType());
            stmt.setString(1, String.valueOf(context.userId()));
            stmt.setString(2, context.userType());
            stmt.setInt(3, context.length());

            session.logger().info(stmt.toString());

            List<AddressBuilder> result = new ArrayList<>(context.length());
            try (ResultSet resultSet = stmt.executeQuery()) {
                while (resultSet.next()) {
                    MarketAddressBuilder builder = MarketAddressBuilder.build(context.passportUser(), resultSet);
                    result.add(builder);
                }
                //found += writeResults(resultSet, false, user, session, writer);
            }

            context.session().logger().info("Market completed");
            callback.completed(result);
        } catch (SQLException e) {
            context.session().logger().info("Market completed");
            callback.failed(e);
        }
    }

    @Override
    public void get(
        final AddressContext context,
        final AddressId id,
        final FutureCallback<AddressBuilder> callback)
    {
        ProxySession session = context.session();
        try (Connection connection = marketConnectionPool.getConnection(session.logger());
             PreparedStatement stmt = connection.prepareStatement(GET_MARKET_ADDRESS)) {
            //stmt.setString(1, user.userType());
            stmt.setString(1, id.addressId());
            session.logger().info(stmt.toString());
            try (ResultSet resultSet = stmt.executeQuery()) {
                boolean next = resultSet.next();
                if (next) {
                    callback.completed(MarketAddressBuilder.build(context.userType(), context.userId(), resultSet));
                } else {
                    callback.failed(new ServerException(HttpStatus.SC_NOT_FOUND, "not found"));
                }
            }
        } catch (Exception e) {
            session.logger().log(Level.SEVERE, "Request failed", e);
            callback.failed(new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, "Database request failure"));
            return;
        }
    }

    @Override
    public void update(
        final AddressContext context,
        final PassportAddressBuilder address,
        final FutureCallback<PassportAddressBuilder> callback)
    {

    }

    @Override
    public void delete(
        final AddressContext context,
        final AddressId id,
        final FutureCallback<Object> callback)
    {
        ProxySession session  = context.session();
        try (Connection connection = marketConnectionPool.getConnection(session.logger());
             PreparedStatement stmt = connection.prepareStatement(DELETE_MARKET_ADDRESS))
        {
            //stmt.setString(1, user.userType());
            stmt.setString(1, id.addressId());
            session.logger().info(stmt.toString());
            try (ResultSet resultSet = stmt.executeQuery()) {
                boolean next = resultSet.next();
                if (next) {
                    callback.completed(id);
                } else {
                    callback.failed(new NotFoundException("Id was not found " + id));
                }
            }
        } catch (Exception e) {
            session.logger().log(Level.SEVERE, "Request failed", e);
            callback.failed(new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, "Database request failure", e));
            return;
        }
    }
}
