package ru.yandex.passport.storage;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.passport.AddressContext;
import ru.yandex.passport.AddressProxy;
import ru.yandex.passport.AddressStorage;
import ru.yandex.passport.address.AddressBuilder;
import ru.yandex.passport.address.AddressId;
import ru.yandex.passport.address.AddressService;
import ru.yandex.passport.address.AsyncGeocoderClient;
import ru.yandex.passport.address.ListAddressContext;
import ru.yandex.passport.address.PassportAddressBuilder;

public class PassportDeliveryDataSyncStorage implements AddressStorage<AddressBuilder> {
    private final HttpHost host;
    private final AsyncClient dataSyncClient;
    private final AsyncGeocoderClient geocoderClient;

    public PassportDeliveryDataSyncStorage(final AddressProxy proxy) {
        this.dataSyncClient = proxy.deliveryDatasyncClient();
        this.host = proxy.config().deliveryDatasyncConfig().host();
        this.geocoderClient = proxy.geocoderClient();
    }

    @Override
    public boolean storeNewValues() {
        return true;
    }

    @Override
    public void list(final ListAddressContext context, final FutureCallback<List<AddressBuilder>> callback) {
        dataSyncClient.execute(
            host,
            new BasicAsyncRequestProducerGenerator(
                "/v1/" + context.userId()
                    + "/personality/profile/market/delivery_addresses?client_id=market&client_name=market"),
            JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING_OK,
            context.session().listener().createContextGeneratorFor(dataSyncClient),
            new DataSyncListCallback(callback, context));
    }

    @Override
    public void get(final AddressContext context, final AddressId id, final FutureCallback<AddressBuilder> callback) {
        dataSyncClient.execute(
            host,
            new BasicAsyncRequestProducerGenerator(
                "/v1/" + context.userId() + "/personality/profile/market/delivery_addresses/" + id.addressId()
                    + "/?client_id=market&client_name=market"),
            JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING_OK,
            context.session().listener().createContextGeneratorFor(dataSyncClient),
            new DataSyncGetCallback(callback, context));
    }

    @Override
    public void create(
        final AddressContext context,
        final PassportAddressBuilder address,
        final FutureCallback<PassportAddressBuilder> callback)
    {
        update(context, address, callback);
    }

    @Override
    public void update(
        final AddressContext context,
        final PassportAddressBuilder address,
        final FutureCallback<PassportAddressBuilder> callback)
    {
        String serialized = JsonType.NORMAL.toString(toDataSyncAddress(address));
        context.session().logger().info("Saving delivery " + serialized);
        try {
            dataSyncClient.execute(
                host,
                new BasicAsyncRequestProducerGenerator(
                    "/v1/" + context.userId()
                    + "/personality/profile/market/delivery_addresses/"
                    + address.id().addressId()
                    + "/?client_id=market&client_name=market",
                    new NByteArrayEntityGenerator(
                        new NStringEntity(
                            serialized,
                            ContentType.APPLICATION_JSON.withCharset(
                                StandardCharsets.UTF_8))),
                    "PUT"),
                JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING_OK,
                context.session().listener()
                    .createContextGeneratorFor(dataSyncClient),
                new DataSyncCreateCallback(callback, address, context));
        } catch (IOException e) {
            callback.failed(e);
        }
    }

    @Override
    public void delete(
        final AddressContext context,
        final AddressId id,
        final FutureCallback<Object> callback)
    {
        BasicAsyncRequestProducerGenerator generator = new BasicAsyncRequestProducerGenerator(
            "/v1/" + context.userId() + "/personality/profile/market/delivery_addresses/"
                + id.addressId()
                + "/?client_id=market&client_name=market",
            null,
            "DELETE");

        dataSyncClient.execute(
            host,
            generator,
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.ANY_GOOD,
                JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING),
            context.session().listener().createContextGeneratorFor(dataSyncClient),
            callback);
    }

    private class DataSyncGetCallback extends AbstractFilterFutureCallback<JsonObject, AddressBuilder> {
        private final AddressContext context;

        public DataSyncGetCallback(
            final FutureCallback<? super AddressBuilder> callback,
            final AddressContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObject) {
            try {
                JsonMap map = resultObject.asMap();
                PassportAddressBuilder addressBuilder =
                    parseFromDataSync(map, context.userType(), context.userId());

                geocoderClient.fillAddress(addressBuilder, callback);
            } catch (Exception e) {
                failed(e);
            }
        }
    }

    private class DataSyncListCallback extends AbstractFilterFutureCallback<JsonObject, List<AddressBuilder>> {
        private final AddressContext context;

        public DataSyncListCallback(
            final FutureCallback<? super List<AddressBuilder>> callback,
            final AddressContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObject) {
            try {
                context.session().logger().info("Delivery completed");
                JsonList items = resultObject.asMap().getList("items");
                if (items.size() == 0) {
                    callback.completed(Collections.emptyList());
                    return;
                }

                MultiFutureCallback<AddressBuilder> mfcb = new MultiFutureCallback<>(callback);
                for (JsonObject itemObj: items) {
                    JsonMap map = itemObj.asMap();
                    PassportAddressBuilder addressBuilder =
                        parseFromDataSync(map, context.userType(), context.userId());
                    if (addressBuilder != null) {
                        geocoderClient.fillAddress(addressBuilder, mfcb.newCallback());
                    }
                }

                mfcb.done();
            } catch (Exception e) {
                failed(e);
            }
        }
    }

    private static class DataSyncCreateCallback extends AbstractFilterFutureCallback<JsonObject, PassportAddressBuilder> {
        private final AddressContext context;
        private final PassportAddressBuilder address;

        public DataSyncCreateCallback(
            final FutureCallback<? super PassportAddressBuilder> callback,
            final PassportAddressBuilder address,
            final AddressContext context)
        {
            super(callback);
            this.context = context;
            this.address = address;
        }

        @Override
        public void completed(final JsonObject resultObject) {
            try {
                context.session().logger().info("SAve returned " + JsonType.NORMAL.toString(resultObject));
                callback.completed(address);
            } catch (Exception e) {
                failed(e);
            }
        }
    }

    public static PassportAddressBuilder parseFromDataSync(
        final JsonMap map,
        final String userType,
        final Object userId)
        throws JsonException
    {
        PassportAddressBuilder builder = new PassportAddressBuilder();
        builder.setMarketAddressLine(map.getString("addressLine", null));
        builder.setCountry(map.getString("country", null));
        builder.setCity(map.getString("city", null));
        builder.setCity(map.getString("street", null));
        builder.setBuilding(map.getString("building", null));

        String id = map.getString("id", null);
        if (!builder.validForGeocoder()) {
            return null;
        }

        if (id == null) {
            return null;
        }

        builder.setId(new AddressId(userType, userId, AddressService.PASSPORT_DELIVERY, id));
        builder.setZip(map.getString("zip", null));
        builder.setRegionId(map.getInt("regionId", null));
        builder.setMetro(map.getString("metro", null));
        builder.setEntrance(map.getString("entrance", null));
        builder.setRoom(map.getString("flat", null));
        builder.setFloor(map.getString("floor", null));
        builder.setIntercom(map.getString("intercom", null));
        builder.setCargoLift(map.getBoolean("cargolift", null));
        builder.setOwnerService(builder.id().ownerService());
        return builder;
    }

    public static JsonMap toDataSyncAddress(final PassportAddressBuilder address) {
        JsonMap map = new JsonMap(BasicContainerFactory.INSTANCE);

        map.put("addressLine", new JsonString(address.addressLine()));
        map.put("country", new JsonString(address.country()));
        map.put("city", new JsonString(address.city()));
        map.put("street", new JsonString(address.street()));
        map.put("building", new JsonString(address.building()));

        map.put("id", new JsonString(address.id().addressId()));

        map.put("zip", new JsonString(address.zip()));
        map.put("regionId", new JsonLong(address.regionId()));
        map.put("metro", new JsonString(address.metro()));
        map.put("entrance", new JsonString(address.entrance()));
        map.put("flat", new JsonString(address.room()));
        map.put("floor", new JsonString(address.floor()));
        map.put("intercom", new JsonString(address.intercom()));
        map.put("cargolift", JsonBoolean.valueOf(address.cargoLift()));
        return map;
    }
}
