package ru.yandex.passport.storage;

import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.Tuple;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.ServerException;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.passport.AddressContext;
import ru.yandex.passport.AddressProxy;
import ru.yandex.passport.AddressStorage;
import ru.yandex.passport.address.AddressBuilder;
import ru.yandex.passport.address.AddressId;
import ru.yandex.passport.address.AddressService;
import ru.yandex.passport.address.GeocoderAddressConverter;
import ru.yandex.passport.address.ListAddressContext;
import ru.yandex.passport.address.Locale;
import ru.yandex.passport.address.PassportAddressBuilder;
import ru.yandex.passport.address.handlers.AbstractAddressHandler;

public class PassportPgStorage implements AddressStorage<AddressBuilder> {
    public static final String PASSPORT_LIST_ADDRESSES =
        "SELECT DISTINCT ON (a.region_id, a.country, a.city, a.street, a.building, a.floor, a.room, a.entrance, a.intercom, a.zip) "
            + AbstractAddressHandler.PASSPORT_COLUMNS_STMT +
            " FROM passport_address as a " +
            "WHERE a.user_id=? AND a.user_type=? AND owner_service=? AND a.locale=? " +
            "AND NOT a.deleted AND NOT a.draft LIMIT ?";
    public static final String PASSPORT_LIST_ADDRESSES_VERTX =
        "SELECT DISTINCT ON (a.region_id, a.country, a.city, a.street, a.building, a.floor, a.room, a.entrance, a.intercom, a.zip) "
            + AbstractAddressHandler.PASSPORT_COLUMNS_STMT +
            " FROM passport_address as a " +
            "WHERE a.user_id=$1 AND a.user_type=$2 AND a.locale=$3 " +
            "AND NOT a.deleted AND NOT a.draft AND owner_service IN ";
//    public static final SqlQuery PASSPORT_LIST_ADDRESSES_QUERY
//        = new SqlQuery("passp_list_address", PASSPORT_LIST_ADDRESSES_VERTX);

    public static final String GET_PASSPORT_ADDRESS =
        "SELECT " + AbstractAddressHandler.PASSPORT_COLUMNS_STMT +
            " FROM passport_address as a " +
            "WHERE a.object_key=? AND a.locale=? AND a.user_id=? AND a.user_type=? " +
            "AND NOT a.deleted AND NOT a.draft";
    public static final String GET_PASSPORT_ADDRESS_VERTX =
        "SELECT " + AbstractAddressHandler.PASSPORT_COLUMNS_STMT +
            " FROM passport_address as a " +
            "WHERE a.object_key=$1 AND a.locale=$2 AND a.user_id=$3 AND a.user_type=$4 " +
            "AND NOT a.deleted AND NOT a.draft";
    public static final SqlQuery GET_PASSPORT_ADDRESS_QUERY =
        new SqlQuery("get_pssp_address", GET_PASSPORT_ADDRESS_VERTX);

//    private static final String LOAD_HIDDEN_ADDRESSES =
//        "SELECT address_id FROM passport_hidden_address " +
//            "WHERE user_id=? AND user_type=? AND hidden_by_service=? LIMIT 300";

    private static final SqlQuery LOAD_HIDDEN_ADDRESSES_VERTX = new SqlQuery("hidden_address",
        "SELECT address_id FROM passport_hidden_address " +
            "WHERE user_id=$1 AND user_type=$2 AND hidden_by_service=$3 LIMIT 300");
//    private static final String DELETE_PASSPORT_ADDRESS =
//        "UPDATE passport_address as a SET deleted = true " +
//            "WHERE a.object_key=? RETURNING id";

    private static final SqlQuery DELETE_PASSPORT_ADDRESS_VERTX =
        new SqlQuery(
            "delete_address",
            "UPDATE passport_address as a SET deleted = true " +
                "WHERE a.object_key=$1 AND a.user_id=$2 AND a.user_type=$3 RETURNING id");

//    private static final String UPDATE_ADDRESS =
//        "UPDATE passport_address SET user_type = ?, user_id = ?, region_id = ?, precise_region_id = " +
//            "?, country = ?, city = ?, street = ?, building = ?, floor =? , " +
//            "room = ?, entrance = ?, intercom = ?, zip = ?, source = ?, comment = ?, latitude = ?, longitude = ?, " +
//            "district = ?, platform = ?, owner_service = ?, subtype = ?, " +
//            "format_version = ?, label = ?, brand_name = ?, phone_id = ?, last_touched_time = ?, " +
//            "version = ?, comment_courier = ?, locale = ?, geocoder_name = ?, geocoder_exact = ?, " +
//            "geocoder_description = ?, geocoder_object_type = ?, org_id = ?, name = ?, uri = ?," +
//            "modification_time = ?, is_portal_uid = ?, draft = ? " +
//            "WHERE object_key = ? AND user_id = ? AND user_type = ? RETURNING *";
    private static final SqlQuery UPDATE_ADDRESS_VERTX =
        new SqlQuery(
            "update_address",
        "UPDATE passport_address SET user_type = $1, user_id = $2, region_id = $3, precise_region_id = " +
            "$4, country = $5, city = $6, street = $7, building = $8, floor = $9 , " +
            "room = $10, entrance = $11, intercom = $12, zip = $13, source = $14, comment = $15, latitude = $16, longitude = $17, " +
            "district = $18, platform = $19, owner_service = $20, subtype = $21, " +
            "format_version = $22, label = $23, brand_name = $24, phone_id = $25, last_touched_time = $26, " +
            "version = $27, comment_courier = $28, locale = $29, geocoder_name = $30, geocoder_exact = $31, " +
            "geocoder_description = $32, geocoder_object_type = $33, org_id = $34, name = $35, uri = $36," +
            "modification_time = $37, is_portal_uid = $38, draft = $39 " +
            "WHERE object_key = $40 AND user_id = $41 AND user_type = $42 RETURNING *");
//    private static final String UPSERT_ADDRESS =
//        "INSERT INTO passport_address(user_type,user_id,region_id,precise_region_id," +
//            "country,city,street,building,floor,room,entrance,intercom,zip,source,comment,latitude,longitude, " +
//            "district,platform,owner_service,subtype,format_version,label,brand_name,phone_id,last_touched_time, " +
//            "version,comment_courier,locale,geocoder_name,geocoder_exact,geocoder_description,geocoder_object_type," +
//            "org_id,name,uri,modification_time,is_portal_uid,draft,object_key) " +
//            "VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)" +
//            " ON CONFLICT (object_key) DO UPDATE SET user_type = ?, user_id = ?, region_id = ?, precise_region_id = " +
//            "?, country = ?, city = ?, street = ?, building = ?, floor =? , " +
//            "room = ?, entrance = ?, intercom = ?, zip = ?, source = ?, comment = ?, latitude = ?, longitude = ?, " +
//            "district = ?, platform = ?, owner_service = ?, subtype = ?, " +
//            "format_version = ?, label = ?, brand_name = ?, phone_id = ?, last_touched_time = ?, " +
//            "version = ?, comment_courier = ?, locale = ?, geocoder_name = ?, geocoder_exact = ?, " +
//            "geocoder_description = ?, geocoder_object_type = ?, org_id = ?, name = ?, uri = ?," +
//            "modification_time = ?, is_portal_uid = ?, draft=?, object_key=? RETURNING *";
    private static final SqlQuery UPSERT_ADDRESS_VERTX =
        new SqlQuery(
            "upsert_address",
        "INSERT INTO passport_address(user_type,user_id,region_id,precise_region_id," +
            "country,city,street,building,floor,room,entrance,intercom,zip,source,comment,latitude,longitude, " +
            "district,platform,owner_service,subtype,format_version,label,brand_name,phone_id,last_touched_time, " +
            "version,comment_courier,locale,geocoder_name,geocoder_exact,geocoder_description,geocoder_object_type," +
            "org_id,name,uri,modification_time,is_portal_uid,draft,object_key) " +
            "VALUES ($1,$2,$3,$4,$5,$6,$7,$8,$9,$10,$11,$12,$13,$14,$15,$16,$17,$18,$19,$20,$21,$22,$23,$24,$25,$26,$27,$28,$29,$30,$31,$32,$33,$34,$35,$36,$37,$38,$39,$40)" +
            " ON CONFLICT (object_key) DO UPDATE SET user_type = $41, user_id = $42, region_id = $43, precise_region_id = " +
            "$44, country = $45, city = $46, street = $47, building = $48, floor = $49 , " +
            "room = $50, entrance = $51, intercom = $52, zip = $53, source = $54, comment = $55, latitude = $56, longitude = $57, " +
            "district = $58, platform = $59, owner_service = $60, subtype = $61, " +
            "format_version = $62, label = $63, brand_name = $64, phone_id = $65, last_touched_time = $66, " +
            "version = $67, comment_courier = $68, locale = $69, geocoder_name = $70, geocoder_exact = $71, " +
            "geocoder_description = $72, geocoder_object_type = $73, org_id = $74, name = $75, uri = $76," +
            "modification_time = $77, is_portal_uid = $78, draft=$79, object_key=$80 RETURNING *");

//    private static final String DELETE_PASSPORT_TRANSLATED_ADDRESS =
//        "UPDATE passport_address as a SET deleted = true " +
//            "WHERE a.source_object_key=? RETURNING id";
    private static final String DELETE_PASSPORT_TRANSLATED_ADDRESS_VERTX =
        "UPDATE passport_address as a SET deleted = true " +
            "WHERE a.source_object_key=$1 RETURNING id";

    private static final SqlQuery DELETE_PASSPORT_TRANSLATED_ADDRESS_QUERY
        = new SqlQuery("delete_translated_address", DELETE_PASSPORT_TRANSLATED_ADDRESS_VERTX);

    private final AddressProxy proxy;
    //private final ThreadLocal<DBConnectionPool> updatePool;
    //private final DBConnectionPool passportConnectionPool;
    private final Map<AddressService, SqlQuery> listQueries;

    public PassportPgStorage(
        final AddressProxy proxy,
        final Set<AddressService> services,
        final Map<AddressService, List<AddressService>> allowedReads) throws ConfigException
    {
        this.proxy = proxy;
        //ImmutableAddressPgConfig pgConfig = proxy.config().passportDbConfig();
        //ImmutablePoolConfig updateConfig = new PoolConfigBuilder(pgConfig).poolSize(2).build();
        //this.updatePool = ThreadLocal.withInitial(() -> new DBConnectionPool(pgConfig.url(), updateConfig, proxy.logger()));
        //this.passportConnectionPool = proxy.pool(proxy.config().passportDbConfig());
        LinkedHashMap<AddressService, SqlQuery> listQueries = new LinkedHashMap<>();

        for (AddressService service: services) {
            Set<AddressService> allowedPg = new LinkedHashSet<>(allowedReads.get(service));
            allowedPg.retainAll(services);
            if (allowedPg.isEmpty()) {
                continue;
            }

            StringBuilder sb  = new StringBuilder(PASSPORT_LIST_ADDRESSES_VERTX);
            sb.append('(');
            for (AddressService allowed: allowedPg) {
                sb.append("\'");
                sb.append(allowed.serviceName());
                sb.append("\'");
                sb.append(',');
            }
            sb.setLength(sb.length() - 1);
            sb.append(')');
            sb.append(" LIMIT $4");
            listQueries.put(service, new SqlQuery(service.serviceName() + "_list_query", sb.toString()));
        }

        this.listQueries = Collections.unmodifiableMap(listQueries);
    }

    @Override
    public void deletePassportTranslatedAddress(
        final ProxySession session,
        final AddressId id)
        throws ServerException
    {
        Tuple tuple = Tuple.tuple();
        tuple.addString(id.addressId());

        proxy.pgClient().executeOnMaster(DELETE_PASSPORT_TRANSLATED_ADDRESS_QUERY, tuple, EmptyFutureCallback.INSTANCE);
//        try (Connection connection = updatePool.get().getConnection(session.logger());
//             PreparedStatement stmt = connection.prepareStatement(DELETE_PASSPORT_TRANSLATED_ADDRESS))
//        {
//            stmt.setString(1, id.addressId());
//            session.logger().info(stmt.toString());
//            try (ResultSet resultSet = stmt.executeQuery()) {
//                StringBuilder msg = new StringBuilder();
//                while (resultSet.next()) {
//                    msg.append(resultSet.getLong("id")).append(", ");
//                }
//                session.logger().info("Delete translated address: " + msg);
//            }
//        } catch (Exception e) {
//            session.logger().log(Level.SEVERE, "Delete translated address failed", e);
//            throw new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, "Database request failure");
//        }
    }

    @Override
    public void list(final ListAddressContext context, final FutureCallback<List<AddressBuilder>> callback) {
        SqlQuery query = listQueries.get(context.service());
        if (query == null) {
            context.session().logger().warning("Null list query for " + context.service());
            callback.completed(Collections.emptyList());
            return;
        }

        Tuple tuple = Tuple.tuple();
        tuple.addString(String.valueOf(context.userId()));
        tuple.addString(String.valueOf(context.userType()));
        tuple.addString(context.locale().toString());
        tuple.addInteger(context.length());
        context.session().logger().info("List executed for " + query.query());
        proxy.pgClient().executeOnAny(
            query,
            tuple,
            true,
            context.session().listener(),
            new ListAddressCallback(context, callback));

//        ProxySession session = context.session();
//        try (Connection connection = passportConnectionPool.getConnection(session.logger());
//             PreparedStatement stmt = connection.prepareStatement(PASSPORT_LIST_ADDRESSES))
//        {
//            //stmt.setString(1, user.userType());
//            stmt.setString(1, String.valueOf(context.userId()));
//            stmt.setString(2, context.userType());
//            stmt.setString(3, service.serviceName());
//            stmt.setString(4, context.locale().toString());
//            stmt.setInt(5, context.length());
//
//            session.logger().info(stmt.toString());
//            List<AddressBuilder> result = new ArrayList<>(context.length());
//            try (ResultSet resultSet = stmt.executeQuery()) {
//                while (resultSet.next()) {
//                    PassportAddressBuilder builder = PassportAddressBuilder.build(context.passportUser(), resultSet);
//                    result.add(builder);
//                }
//
//                //found += writeResults(resultSet, false, user, session, writer);
//            }
//            if (result.isEmpty() && context.convert() && !Locale.RU.equals(context.locale())) {
//                GeocoderAddressConverter converter = new GeocoderAddressConverter(proxy);
//                converter.convertList(context, callback, service);
//            } else {
//                context.session().logger().info("Got for " + service.serviceName() + " results " + result.size());
//                callback.completed(result);
//            }
//        } catch (SQLException e) {
//            callback.failed(e);
//        }
    }

    private class ListAddressCallback extends AbstractFilterFutureCallback<RowSet<Row>, List<AddressBuilder>> {
        private final ListAddressContext context;
        private final FutureCallback<List<AddressBuilder>> convertCallback;

        public ListAddressCallback(final ListAddressContext context, final FutureCallback<List<AddressBuilder>> callback) {
            super(callback);

            this.context = context;
            this.convertCallback = callback;
        }

        @Override
        public void completed(final RowSet<Row> rows) {
            List<AddressBuilder> result = new ArrayList<>(context.length());
            try {
                for (Row row: rows) {
                    PassportAddressBuilder builder =
                        PassportAddressBuilder.build(context.passportUser(), row);
                    result.add(builder);
                }

                context.session().logger().info("List completed for " + context.service().serviceName() + " size " + result.size());
                if (result.isEmpty() && context.convert() && !Locale.RU.equals(context.locale())) {
                    GeocoderAddressConverter converter = new GeocoderAddressConverter(proxy);
                    converter.convertList(context, convertCallback, context.service());
                    return;
                }
            } catch (Exception se) {
                failed(se);
                return;
            }

            callback.completed(result);
        }
    }

    @Override
    public void get(
        final AddressContext context,
        final AddressId id,
        final FutureCallback<AddressBuilder> callback)
    {
        Tuple tuple = Tuple.tuple();
        tuple.addString(id.addressId());
        tuple.addString(context.locale().toString());
        tuple.addString(id.userIdString());
        tuple.addString(id.userType());

        proxy.pgClient().executeOnAny(
            GET_PASSPORT_ADDRESS_QUERY,
            tuple,
            true,
            context.session().listener(),
            new GetCallback(callback, id, context));

//        ProxySession session = context.session();
//        try (Connection connection = passportConnectionPool.getConnection(session.logger());
//             PreparedStatement stmt = connection.prepareStatement(GET_PASSPORT_ADDRESS))
//        {
//            //stmt.setString(1, user.userType());
//            stmt.setString(1, id.addressId());
//            stmt.setString(2, context.locale().toString());
//            stmt.setString(3, id.userIdString());
//            stmt.setString(4, id.userType());
//
//            session.logger().info(stmt.toString());
//            try (ResultSet resultSet = stmt.executeQuery()) {
//                boolean next = resultSet.next();
//                if (next) {
//                    callback.completed(PassportAddressBuilder.build(context.userType(), context.userId(), resultSet));
//                } else if (context.convert() && !Locale.RU.equals(context.locale())) {
//                    GeocoderAddressConverter converter = new GeocoderAddressConverter(proxy);
//                    converter.convert(context, callback, id);
//                } else {
//                    callback.failed(new ServerException(HttpStatus.SC_NOT_FOUND, "not found"));
//                }
//            }
//        } catch (Exception e) {
//            session.logger().log(Level.SEVERE, "Request failed", e);
//            callback.failed(new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, "Database request failure"));
//        }
    }

    private class GetCallback extends AbstractFilterFutureCallback<RowSet<Row>, AddressBuilder> {
        private final AddressId id;
        private final AddressContext context;
        private final FutureCallback<AddressBuilder> convertCallback;

        public GetCallback(
            final FutureCallback<AddressBuilder> callback,
            final AddressId id,
            final AddressContext context)
        {
            super(callback);
            this.convertCallback = callback;
            this.context = context;
            this.id = id;
        }

        @Override
        public void completed(final RowSet<Row> rows) {
            try {
                boolean convert = context.convert() && !Locale.RU.equals(context.locale());
                if (rows.rowCount() <= 0) {
                    if (!convert) {
                        callback.failed(new ServerException(HttpStatus.SC_NOT_FOUND, "not found"));
                    } else {
                        GeocoderAddressConverter converter = new GeocoderAddressConverter(proxy);
                        converter.convert(context, convertCallback, id);
                    }

                    return;
                }

                Row row = rows.iterator().next();
                callback.completed(PassportAddressBuilder.build(context.userType(), context.userId(), row));
            } catch (Exception se) {
                callback.failed(se);
            }
        }
    }

    @Override
    public void hidden(final AddressContext context, final FutureCallback<Set<AddressId>> callback) {
        Tuple tuple = Tuple.tuple();
        tuple.addString(String.valueOf(context.userId()));
        tuple.addString(context.userType());
        tuple.addString(context.serviceName());

        proxy.pgClient().executeOnMaster(
            LOAD_HIDDEN_ADDRESSES_VERTX,
            tuple,
            context.session().listener(),
            new HiddenAddressesCallback(callback));

//        ProxySession session = context.session();
//        try (Connection connection = passportConnectionPool.getConnection(session.logger());
//             PreparedStatement stmt = connection.prepareStatement(LOAD_HIDDEN_ADDRESSES))
//        {
//
//            session.logger().info(stmt.toString());
//            //stmt.setString(1, user.userType());
//            stmt.setString(1, String.valueOf(context.userId()));
//            stmt.setString(2, context.userType());
//            stmt.setString(3, context.serviceName());
//
//
//            try (ResultSet resultSet = stmt.executeQuery()) {
//                while (resultSet.next()) {
//                    result.add();
//                }
//            }
//
//            callback.completed(result);
//        } catch (SQLException | IOException e) {
//            callback.failed(e);
//        }
    }

    private static class HiddenAddressesCallback extends AbstractFilterFutureCallback<RowSet<Row>, Set<AddressId>> {
        public HiddenAddressesCallback(final FutureCallback<? super Set<AddressId>> callback) {
            super(callback);
        }

        @Override
        public void completed(final RowSet<Row> rows) {
            Set<AddressId> result = new LinkedHashSet<>();
            try {
                for (Row row: rows) {
                    result.add(AddressId.parseOuterId(row.getString("address_id")));
                }
            } catch (Exception e) {
                failed(e);
                return;
            }

            callback.completed(result);
        }
    }

    @Override
    public void delete(AddressContext context, AddressId id, FutureCallback<Object> callback) {
        Tuple tuple = Tuple.tuple();
        tuple.addString(id.addressId());
        tuple.addString(String.valueOf(context.userId()));
        tuple.addString(context.userType());

        proxy.pgClient().executeOnMaster(
            DELETE_PASSPORT_ADDRESS_VERTX,
            tuple,
            context.session().listener(),
            new DeleteCallback(callback, id));

//        try (Connection connection = passportConnectionPool.getConnection(session.logger());
//             PreparedStatement stmt = connection.prepareStatement(DELETE_PASSPORT_ADDRESS))
//        {
//            //stmt.setString(1, user.userType());
//            stmt.setString(1, id.addressId());
//            session.logger().info(stmt.toString());
//            try (ResultSet resultSet = stmt.executeQuery()) {
//                boolean next = resultSet.next();
//                if (next) {
//                    callback.completed(id);
//                } else {
//                    callback.failed(new NotFoundException("Id was not found " + id));
//                }
//            }
//        } catch (Exception e) {
//            session.logger().log(Level.SEVERE, "Request failed", e);
//            callback.failed(new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, "Database request failure", e));
//            return;
//        }
    }

    private static class DeleteCallback extends AbstractFilterFutureCallback<RowSet<Row>, Object> {
        private final AddressId id;

        public DeleteCallback(final FutureCallback<? super Object> callback, final AddressId id) {
            super(callback);
            this.id = id;
        }

        @Override
        public void completed(final RowSet<Row> rows) {
            try {
                if (rows.rowCount() <= 0) {
                    callback.failed(new ServerException(HttpStatus.SC_NOT_FOUND, "not found"));
                    return;
                }

                callback.completed(id);
            } catch (Exception se) {
                callback.failed(se);
            }
        }
    }

    private void upsert(
        final AddressContext context,
        final PassportAddressBuilder address,
        final FutureCallback<PassportAddressBuilder> callback)
    {
        Tuple tuple = Tuple.tuple();

        OffsetDateTime modificationTime = Objects.requireNonNullElseGet(
            address.modificationTime(),
            OffsetDateTime::now);
        int index = fillStatement(tuple, context, address, modificationTime, 0);
        fillStatement(tuple, context, address, modificationTime, index);
        address.setModificationTime(modificationTime);

        proxy.pgClient().executeOnMaster(
            UPSERT_ADDRESS_VERTX,
            tuple,
            context.session().listener(),
            new UpsertCallback(callback, context, address.id()));

//        ProxySession session = context.session();
//        try (Connection connection = updatePool.get().getConnection(
//            context.session().logger());
//             PreparedStatement stmt = connection.prepareStatement(UPSERT_ADDRESS))
//        {
//            //stmt.setString(1, user.userType());
//            OffsetDateTime modificationTime = Objects.requireNonNullElseGet(
//                address.modificationTime(),
//                OffsetDateTime::now);
//            int index = fillStatement(stmt, context, address, modificationTime, 0);
//            fillStatement(stmt, context, address, modificationTime, index);
//            address.setModificationTime(modificationTime);
//
//            context.session().logger().info(stmt.toString());
//            context.session().logger().info(address.toString());
//            try (ResultSet resultSet = stmt.executeQuery()) {
//                if (!resultSet.next()) {
//                    callback.failed(new NotFoundException("Id was not found " + address.id()));
//                    return;
//                }
//                Long id = resultSet.getLong("id");
//                //resultSet.get();
//                PassportAddressBuilder updated = PassportAddressBuilder.build(
//                    context.userType(),
//                    context.userId(),
//                    resultSet);
//                context.session().logger().info("Saved " + id);
//
//                updated.setId(address.id());
//                callback.completed(updated);
//            }
//        } catch (Exception e) {
//            session.logger().log(Level.WARNING, "Failed to update record", e);
//            callback.failed(
//                new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, "Failed to update address " + address.id(), e));
//            return;
//        }
    }

    private static class UpsertCallback extends AbstractFilterFutureCallback<RowSet<Row>, PassportAddressBuilder> {
        private final AddressContext context;
        private final AddressId addressId;

        public UpsertCallback(
            final FutureCallback<? super PassportAddressBuilder> callback,
            final AddressContext context,
            final AddressId addressId)
        {
            super(callback);
            this.context = context;
            this.addressId = addressId;
        }

        @Override
        public void completed(final RowSet<Row> rows) {
            if (rows.rowCount() <= 0) {
                callback.failed(new NotFoundException("Not found"));
                return;
            }

            Row row = rows.iterator().next();
            Long id = row.getLong("id");
            //resultSet.get();
            PassportAddressBuilder updated = PassportAddressBuilder.build(
                context.userType(),
                context.userId(),
                row);
            context.session().logger().info("Saved " + id);

            updated.setId(addressId);
            callback.completed(updated);
        }
    }
    private int fillStatement(
        final Tuple stmt,
        final AddressContext context,
        final PassportAddressBuilder address,
        final OffsetDateTime modificationTime,
        final int startIndex)
    {
        stmt.addString(context.userType());
        stmt.addString(String.valueOf(context.userId()));
        stmt.addInteger(address.regionId());
        stmt.addInteger(address.preciseRegionId());
        stmt.addString(address.country());
        stmt.addString(address.city());
        stmt.addString(address.street());
        stmt.addString(address.building());
        stmt.addString(address.floor());
        stmt.addString(address.room());
        stmt.addString(address.entrance());
        stmt.addString(address.intercom());
        stmt.addString(address.zip());
        stmt.addString(address.subtype());
        stmt.addString(address.comment());
        if (address.location() != null && address.location().getLatitude() != null) {
            stmt.addBigDecimal(address.location().getLatitude());
        } else {
            stmt.addBigDecimal(null);
        }
        if (address.location() != null && address.location().getLongitude() != null) {
            stmt.addBigDecimal(address.location().getLongitude());
        } else {
            stmt.addBigDecimal(null);
        }
        stmt.addString(address.district());
        stmt.addString(address.platform());
        stmt.addString(address.ownerService().serviceName());
        stmt.addString(address.subtype());

        stmt.addString(address.formatVersion());
        stmt.addString(address.label());
        stmt.addString(address.brandName());
        stmt.addString(address.phoneId());
        if (address.lastTouchedTime() != null) {
            stmt.addLocalDateTime(address.lastTouchedTime().toLocalDateTime());
        } else {
            stmt.addLocalDateTime(null);
        }

        stmt.addInteger(address.version());
        stmt.addString(address.commentCourier());
        stmt.addString(address.locale());
        stmt.addString(address.geocoderName());
        stmt.addBoolean(address.geocoderExact());
        stmt.addString(address.geocoderDescription());
        stmt.addString(address.geocoderObjectType());
        stmt.addLong(address.orgId());
        stmt.addString(address.name());
        stmt.addString(address.uri());

        if (modificationTime != null) {
            stmt.addLocalDateTime(modificationTime.toLocalDateTime());
        } else {
            stmt.addLocalDateTime(null);
        }

        stmt.addBoolean(address.isPortalUid());
        stmt.addBoolean(address.draft());

        stmt.addString(address.id().addressId());

        return startIndex + 40;
    }

//    private int fillStatement(
//        final PreparedStatement stmt,
//        final AddressContext context,
//        final PassportAddressBuilder address,
//        final OffsetDateTime modificationTime,
//        final int startIndex)
//        throws SQLException
//    {
//        stmt.setString(startIndex + 1, context.userType());
//        stmt.setLong(startIndex + 2, context.userIdLong());
//        stmt.setObject(startIndex + 3, address.regionId());
//        stmt.setObject(startIndex + 4, address.preciseRegionId());
//        stmt.setString(startIndex + 5, address.country());
//        stmt.setString(startIndex + 6, address.city());
//        stmt.setString(startIndex + 7, address.street());
//        stmt.setString(startIndex + 8, address.building());
//        stmt.setString(startIndex + 9, address.floor());
//        stmt.setString(startIndex + 10, address.room());
//        stmt.setString(startIndex + 11, address.entrance());
//        stmt.setString(startIndex + 12, address.intercom());
//        stmt.setString(startIndex + 13, address.zip());
//        stmt.setString(startIndex + 14, address.subtype());
//        stmt.setString(startIndex + 15, address.comment());
//        if (address.location() != null && address.location().getLatitude() != null) {
//            stmt.setObject(startIndex + 16, address.location().getLatitude().doubleValue());
//        } else {
//            stmt.setObject(startIndex + 16, null);
//        }
//        if (address.location() != null && address.location().getLongitude() != null) {
//            stmt.setObject(startIndex + 17, address.location().getLongitude().doubleValue());
//        } else {
//            stmt.setObject(startIndex + 17, null);
//        }
//        stmt.setString(startIndex + 18, address.district());
//        stmt.setString(startIndex + 19, address.platform());
//        stmt.setString(startIndex + 20, address.ownerService().serviceName());
//        stmt.setString(startIndex + 21, address.subtype());
//
//        stmt.setString(startIndex + 22, address.formatVersion());
//        stmt.setString(startIndex + 23, address.label());
//        stmt.setString(startIndex + 24, address.brandName());
//        stmt.setString(startIndex + 25, address.phoneId());
//        stmt.setObject(startIndex + 26, address.lastTouchedTime());
//        stmt.setObject(startIndex + 27, address.version());
//        stmt.setString(startIndex + 28, address.commentCourier());
//        stmt.setString(startIndex + 29, address.locale());
//        stmt.setString(startIndex + 30, address.geocoderName());
//        stmt.setObject(startIndex + 31, address.geocoderExact());
//        stmt.setString(startIndex + 32, address.geocoderDescription());
//        stmt.setString(startIndex + 33, address.geocoderObjectType());
//        stmt.setObject(startIndex + 34, address.orgId());
//        stmt.setString(startIndex + 35, address.name());
//        stmt.setString(startIndex + 36, address.uri());
//
//        stmt.setObject(startIndex + 37, modificationTime);
//        stmt.setBoolean(startIndex + 38, address.isPortalUid());
//        stmt.setBoolean(startIndex + 39, address.draft());
//
//        stmt.setString(startIndex + 40, address.id().addressId());
//
//        return startIndex + 40;
//    }

    @Override
    public void update(
        final AddressContext context,
        final PassportAddressBuilder address,
        final FutureCallback<PassportAddressBuilder> callback)
    {
        ProxySession session = context.session();
        try {
            if (session.params().getBoolean("insert", false)) {
                upsert(context, address, callback);
                return;
            }
        } catch (BadRequestException bre) {
            callback.failed(bre);
            return;
        }

        Tuple tuple = Tuple.tuple();

        OffsetDateTime modificationTime = Objects.requireNonNullElseGet(
            address.modificationTime(),
            OffsetDateTime::now);
        fillStatement(tuple, context, address, modificationTime, 0);
        tuple.addString(String.valueOf(context.userId()));
        tuple.addString(context.userType());

        address.setModificationTime(modificationTime);

        proxy.pgClient().executeOnMaster(
            UPDATE_ADDRESS_VERTX,
            tuple,
            context.session().listener(),
            new UpdateCallback(callback, address.id(), context));

//        try (Connection connection = updatePool.get().getConnection(
//            context.session().logger());
//             PreparedStatement stmt = connection.prepareStatement(UPDATE_ADDRESS)) {
//            //stmt.setString(1, user.userType());
//            OffsetDateTime modificationTime = Objects.requireNonNullElseGet(
//                address.modificationTime(),
//                OffsetDateTime::now);
//            int index = fillStatement(stmt, context, address, modificationTime, 0);
//            stmt.setString(index + 1, String.valueOf(context.userId()));
//            stmt.setString(index + 2, context.userType());
//
//            address.setModificationTime(modificationTime);
//
//            //context.session().logger().info(stmt.toString());
//            //context.session().logger().info(address.toString());
//            try (ResultSet resultSet = stmt.executeQuery()) {
//                if (!resultSet.next()) {
//                    callback.failed(new NotFoundException("Id was not found " + address.id()));
//                    return;
//                }
//                Long id = resultSet.getLong("id");
//                //resultSet.get();
//                PassportAddressBuilder updated = PassportAddressBuilder.build(
//                    context.userType(),
//                    context.userId(),
//                    resultSet);
//                context.session().logger().info("Saved " + id);
//
//                updated.setId(address.id());
//                callback.completed(updated);
//            }
//            context.session().logger().info("Returning connection to pool");
//        } catch (Exception e) {
//            session.logger().log(Level.WARNING, "Failed to update record", e);
//            callback.failed(
//                new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, "Failed to update address " + address.id(), e));
//            return;
//        }
    }

    private static class UpdateCallback extends AbstractFilterFutureCallback<RowSet<Row>, PassportAddressBuilder> {
        private final AddressId addressId;
        private final AddressContext context;

        public UpdateCallback(
            final FutureCallback<? super PassportAddressBuilder> callback,
            final AddressId addressId,
            final AddressContext context)
        {
            super(callback);
            this.addressId = addressId;
            this.context = context;
        }

        @Override
        public void completed(final RowSet<Row> rows) {
            if (rows.rowCount() <= 0) {
                callback.failed(new NotFoundException("Id was not found " + addressId));
                return;
            }
            Row resultSet = rows.iterator().next();
            Long id = resultSet.getLong("id");
            //resultSet.get();
            PassportAddressBuilder updated = PassportAddressBuilder.build(
                context.userType(),
                context.userId(),
                resultSet);
            context.session().logger().info("Saved " + id);

            updated.setId(addressId);
            callback.completed(updated);
        }
    }

    @Override
    public String name() {
        return listQueries.keySet().toString();
    }
}
