package ru.yandex.passport;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.spec.AlgorithmParameterSpec;

import javax.crypto.Cipher;

import ru.yandex.base64.Base64Decoder;
import ru.yandex.base64.Base64Encoder;
import ru.yandex.function.ByteArrayCopier;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.util.string.StringUtils;

public abstract class AbstractDocumentsStorage implements DocumentsStorage {
    private static final String ENCRYPTION_ALGORITHM = "AES/CBC/PKCS5Padding";
    protected static final long NO_ENCRYPTION_KEY_VERSION = 0;
    protected static final long MANUAL_ENCRYPT_KEY_VERSION = -1;

    private final Key aesKey;
    private final AlgorithmParameterSpec aesIv;

    protected AbstractDocumentsStorage(final Key aesKey, final AlgorithmParameterSpec aesIv) {
        this.aesKey = aesKey;
        this.aesIv = aesIv;
    }

    protected String getDocDataDecrypted(final JsonObject object) throws JsonException, IOException {
        long keyVersion = object.asMap().getLong("key_version", NO_ENCRYPTION_KEY_VERSION);
        String data = object.asMap().getString("doc_data");
        if (keyVersion == MANUAL_ENCRYPT_KEY_VERSION) {
            return decrypt(data);
        } else if (keyVersion == NO_ENCRYPTION_KEY_VERSION) {
            return data;
        } else {
            throw new IllegalStateException("Invalid key version");
        }
    }

    protected String encrypt(final String str) throws IOException {
        try {
            Cipher cipher = Cipher.getInstance(ENCRYPTION_ALGORITHM);
            cipher.init(Cipher.ENCRYPT_MODE, aesKey, aesIv);
            Base64Encoder encoder = new Base64Encoder();
            encoder.process(cipher.doFinal(StringUtils.getUtf8Bytes(str)));
            return encoder.toString();
        } catch (GeneralSecurityException e) {
            throw new IOException(e);
        }
    }

    protected String decrypt(final String str) throws IOException {
        try {
            Cipher cipher = Cipher.getInstance(ENCRYPTION_ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, aesKey, aesIv);
            Base64Decoder decoder = new Base64Decoder();
            decoder.process(str.toCharArray());
            return new String(
                            cipher.doFinal(decoder.processWith(ByteArrayCopier.INSTANCE)),
                            StandardCharsets.UTF_8
                    );
        } catch (GeneralSecurityException e) {
            throw new IOException(e);
        }
    }
}
