package ru.yandex.passport;

import java.io.IOException;
import java.time.OffsetDateTime;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.InputStreamBody;

import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.passport.config.AvatarConfig;
import ru.yandex.passport.document.BaseDocumentBuilder;
import ru.yandex.passport.document.Document;
import ru.yandex.passport.document.DocumentSourceService;
import ru.yandex.passport.document.DocumentType;
import ru.yandex.passport.document.ImageBuilder;

public class DocumentImageDiskCopyHandler extends DocumentsProxyHandler {
    private final DocumentsStorage storage;
    private final AsyncClient avatarClient;
    private final AsyncClient diskClient;
    private final AvatarConfig avatarConfig;
    private final ImmutableHttpHostConfig diskConfig;
    private final String baseUri;

    public DocumentImageDiskCopyHandler(final DocumentsProxy proxy) throws ConfigException {
        super(proxy);
        this.storage = proxy.storage();
        this.diskConfig = proxy.config().disk();
        this.avatarClient = proxy.avatarUploadClient();
        this.avatarConfig = proxy.config().avatar();
        this.baseUri = "/put-" + avatarConfig.namespace() + "/";
        this.diskClient = proxy.client(
            "DiskProxy",
            proxy.config().disk());
    }

    @Override
    public void handle(final ProxySession session, final DocumentSourceService service) throws HttpException, IOException {
        CopyDiskContext context = new CopyDiskContext(session, service);
        QueryConstructor qc = new QueryConstructor("/api/async/passport/disk/document?");
        qc.append("uid", context.userId());
        qc.append("path", context.path());

        BasicAsyncRequestProducerGenerator generator = new BasicAsyncRequestProducerGenerator(qc.toString());
        SaveCallback saveCallback = new SaveCallback(context.session());
        AsyncClient client = diskClient.adjust(context.session().context());
        context.session().logger().info("Request " + diskConfig.host() + " " + qc.toString());
        client.execute(
            diskConfig.host(),
            generator,
            NByteArrayEntityAsyncConsumerFactory.OK,
            new DownloadCallback(saveCallback, context));
    }

    private class DownloadCallback extends AbstractFilterFutureCallback<HttpEntity, Document> {
        private final CopyDiskContext context;

        public DownloadCallback(
            final FutureCallback<? super Document> callback,
            final CopyDiskContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final HttpEntity entity) {
            ProxySession session = context.session();
            QueryConstructor qc = new QueryConstructor(baseUri + UUID.randomUUID());
            AsyncClient client = avatarClient.adjust(session.context());
            BasicAsyncRequestProducerGenerator generator;
            try {
                MultipartEntityBuilder builder =
                    MultipartEntityBuilder.create();
                builder.addPart("file", new InputStreamBody(entity.getContent(), "document.jpg"));
                generator =
                    new BasicAsyncRequestProducerGenerator(qc.toString(), builder.build());
            } catch (IOException ioe) {
                failed(ioe);
                return;
            }
            context.session().logger().info("Downloaded from disk length: " + entity.getContentLength() + " type: " + entity.getContentType() + " " + entity.getClass());
            client.execute(
                avatarConfig.uploadConfig().host(),
                generator,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().adjustContextGenerator(
                    client.httpClientContextGenerator()),
                new AvatarUploadCallback(callback, context));
            context.session().logger().info("Completed");
        }
    }

    private static class CopyDiskContext extends DocumentsContext {
        private final String path;
        private final DocumentType documentType;

        public CopyDiskContext(ProxySession session, DocumentSourceService service) throws BadRequestException {
            super(session, service);

            String path = session.params().getString("path");
            if (path.startsWith("disk:")) {
                path = "/disk" + path.substring(5);
            } else if (path.startsWith("photounlim:")) {
                path = "/photounlim" + path.substring("photounlim:".length());
            }
            this.path = path;
            this.documentType = session.params().getEnum(DocumentType.class, "doc_type", DocumentType.NATIONAL_ID);
        }

        public String path() {
            return path;
        }

        public DocumentType documentType() {
            return documentType;
        }
    }

    private static class SaveCallback extends AbstractDocumentsProxySessionCallback<Document> {
        public SaveCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Document document) {
            try {
                JsonType jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter jw = jsonType.create(sbw)) {
                    jw.value(document);
                }

                session.response(
                    HttpStatus.SC_OK,
                    new StringEntity(sbw.toString(), ContentType.APPLICATION_JSON));
            } catch (IOException | BadRequestException ioe) {
                failed(ioe);
            }
        }
    }

    private class AvatarUploadCallback
        extends AbstractFilterFutureCallback<JsonObject, Document>
    {
        private final CopyDiskContext context;

        public AvatarUploadCallback(
            final FutureCallback<? super Document> callback,
            final CopyDiskContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject avatarObj) {
            try {
                JsonMap map = avatarObj.asMap();
                context.session().logger().info(
                    "Avatar upload result " + JsonType.NORMAL.toString(avatarObj));

                BaseDocumentBuilder builder = new BaseDocumentBuilder();
                builder.images(
                        Stream.concat(
                                builder.images().stream(),
                                Stream.of(new ImageBuilder().fillAvatar(map, avatarConfig))
                        ).collect(Collectors.toList())
                );
                builder.draft(true);
                builder.id(UUID.randomUUID().toString());
                builder.createTime(OffsetDateTime.now());
                builder.docType(context.documentType());
                builder.service(context.service());
                storage.put(context.session(), context.userId(), builder, callback);
            } catch (JsonException je) {
                failed(je);
            }
        }
    }
}
