package ru.yandex.passport;

import java.io.IOException;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.passport.config.AvatarConfig;
import ru.yandex.passport.document.BaseDocumentBuilder;
import ru.yandex.passport.document.Document;
import ru.yandex.passport.document.DocumentSourceService;
import ru.yandex.passport.document.DocumentType;
import ru.yandex.passport.document.ImageBuilder;

public class DocumentImageUploadHandler extends DocumentsProxyHandler {
    private final DocumentsStorage storage;
    private final AsyncClient avatarClient;
    private final AvatarConfig avatarConfig;
    private final String baseUri;
        //private final DocumentsProxy proxy;

    public DocumentImageUploadHandler(final DocumentsProxy proxy) {
        super(proxy);
        this.storage = proxy.storage();
        this.avatarClient = proxy.avatarUploadClient();
        this.avatarConfig = proxy.config().avatar();
        this.baseUri = "/put-" + avatarConfig.namespace() + "/";
    }

    @Override
    public void handle(final ProxySession session, final DocumentSourceService service) throws HttpException, IOException {
        if (!(session.request() instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Missing request body");
        }

        UploadContext uploadContext = new UploadContext(session, service);
        HttpEntity entity = ((HttpEntityEnclosingRequest) session.request()).getEntity();

        QueryConstructor qc = new QueryConstructor(baseUri + UUID.randomUUID().toString());
        BasicAsyncRequestProducerGenerator generator = new BasicAsyncRequestProducerGenerator(qc.toString(), entity);
        generator.addHeader("Content-Disposition", "attachment; filename=\"document.jpg\"");
        generator.copyHeader(session.request(), HttpHeaders.CONTENT_TYPE);

        if (uploadContext.id() != null || uploadContext.originalId() != null) {
            storage.get(
                session,
                uploadContext.userId(),
                uploadContext.id(),
                uploadContext.originalId(),
                new GetCallback(generator, uploadContext));
        } else {
            BaseDocumentBuilder document = uploadContext.documentType().builder();
            document.docType(uploadContext.documentType());
            document.service(service);
            uploadImage(generator, uploadContext, document);
        }
    }

    private static class UploadContext extends DocumentsContext {
        private final String id;
        private final String originalId;
        private final DocumentType documentType;
        private final boolean save;

        public UploadContext(ProxySession session, DocumentSourceService service) throws BadRequestException {
            super(session, service);
            this.id = session.params().getString("id", null);
            this.originalId = session.params().getString("original_id", null);
            this.documentType = session.params().getEnum(DocumentType.class, "doc_type", DocumentType.NATIONAL_ID);
            this.save = session.params().getBoolean("save", true);
        }

        public String id() {
            return id;
        }

        public DocumentType documentType() {
            return documentType;
        }

        public boolean save() {
            return save;
        }

        public String originalId() {
            return originalId;
        }
    }

    private void uploadImage(
        final BasicAsyncRequestProducerGenerator generator,
        final UploadContext context,
        final BaseDocumentBuilder document)
    {
        AsyncClient client = avatarClient.adjust(context.session().context());
        client.execute(
            avatarConfig.uploadConfig().host(),
            generator,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().adjustContextGenerator(
                client.httpClientContextGenerator()),
            new AvatarUploadCallback(
                new SaveCallback(context.session()),
                document,
                context));
    }

    private class GetCallback extends AbstractDocumentsProxySessionCallback<BaseDocumentBuilder> {
        private final BasicAsyncRequestProducerGenerator generator;
        private final UploadContext uploadContext;

        public GetCallback(
            final BasicAsyncRequestProducerGenerator generator,
            final UploadContext uploadContext)
        {
            super(uploadContext.session());
            this.generator = generator;
            this.uploadContext = uploadContext;
        }

        @Override
        public void completed(final BaseDocumentBuilder document) {
            if (document == null) {
                failed(new NotFoundException("Document with id " + uploadContext.id() + " was not found"));
                return;
            }
            uploadImage(generator, uploadContext, document);
        }
    }

    private static class SaveCallback extends AbstractDocumentsProxySessionCallback<Document> {
        public SaveCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Document document) {
            try {
                JsonType jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter jw = jsonType.create(sbw)) {
                    jw.value(document);
                }

                session.response(
                    HttpStatus.SC_OK,
                    new StringEntity(sbw.toString(), ContentType.APPLICATION_JSON));
            } catch (IOException | BadRequestException ioe) {
                failed(ioe);
            }
        }
    }

    private class AvatarUploadCallback
        extends AbstractFilterFutureCallback<JsonObject, Document>
    {
        private final UploadContext context;
        private final BaseDocumentBuilder document;

        public AvatarUploadCallback(
            final FutureCallback<Document> callback,
            final BaseDocumentBuilder document,
            final UploadContext context)
        {
            super(callback);
            this.context = context;
            this.document = document;
        }

        @Override
        public void completed(final JsonObject avatarObj) {
            try {
                JsonMap map = avatarObj.asMap();
                if (!context.save()) {
                    ImageBuilder image = new ImageBuilder().fillAvatar(map, avatarConfig);
                    JsonType jsonType = JsonTypeExtractor.NORMAL.extract(context.session().params());
                    StringBuilderWriter sbw = new StringBuilderWriter();
                    try (JsonWriter jw = jsonType.create(sbw)) {
                        jw.value(image);
                    }

                    context.session().response(
                            HttpStatus.SC_OK,
                            new StringEntity(sbw.toString(), ContentType.APPLICATION_JSON)
                    );
                } else {
                    context.session().logger().info("Avatar upload result " + JsonType.NORMAL.toString(avatarObj));
                    BaseDocumentBuilder document = this.document;
                    document.images(
                            Stream.concat(
                                    document.images().stream(),
                                    Stream.of(new ImageBuilder().fillAvatar(map, avatarConfig))
                            ).collect(Collectors.toList())
                    );
                    if (document.draft() == null) {
                        document.draft(true);
                    }
                    storage.put(context.session(), context.userId(), document, callback);
                }
            } catch (Exception je) {
                failed(je);
            }
        }
    }
}
