package ru.yandex.passport;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;

import ru.yandex.blackbox.BlackboxAttributeType;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.blackbox.BlackboxUserinfos;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.passport.document.Document;
import ru.yandex.passport.document.DocumentSourceService;
import ru.yandex.passport.document.DocumentType;

public class DocumentListHandler extends DocumentsProxyHandler {
    public DocumentListHandler(final DocumentsProxy proxy) {
        super(proxy);
    }

    @Override
    public void handle(final ProxySession session, final DocumentSourceService service) throws HttpException {
        ListContext context = new ListContext(session, service);
        proxy.blackboxClient().userinfo(
                new BlackboxUserinfoRequest(context.userId())
                        .attributes(BlackboxAttributeType.DOCUMENTS_AGREEMENT_ACCEPTED)
                        .addHeader(
                                YandexHeaders.X_YA_SERVICE_TICKET,
                                proxy.blackboxTvm2Ticket()
                        ),
                session.listener().createContextGeneratorFor(proxy.blackboxClient()),
                new BlackboxCallback(context)
        );
    }

    private static class ListContext extends DocumentsContext {
        private final DocumentType documentType;

        public ListContext(ProxySession session, DocumentSourceService service) throws BadRequestException {
            super(session, service);
            documentType = session.params().getEnum(DocumentType.class, "doc_type", null);
        }

        public DocumentType documentType() {
            return documentType;
        }
    }

    private class BlackboxCallback extends AbstractDocumentsProxySessionCallback<BlackboxUserinfos> {
        private final ListContext context;

        BlackboxCallback(ListContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(BlackboxUserinfos blackboxUserinfos) {
            boolean documentsAgreementAccepted = blackboxUserinfos
                    .stream()
                    .findFirst()
                    .flatMap(info -> Optional.ofNullable(
                            info.attributes().get(BlackboxAttributeType.DOCUMENTS_AGREEMENT_ACCEPTED)
                    ))
                    .map(jsonObject -> {
                        try {
                            return jsonObject.asBoolean();
                        } catch (JsonBadCastException e) {
                            proxy.logger().log(Level.WARNING, "Incorrect blackbox response: ", e);
                            return false;
                        }
                    })
                    .orElse(false);

            proxy.storage().list(
                context.session(),
                context.userId(),
                context.documentType(),
                documentsAgreementAccepted
                        ? null
                        : context.service(),
                new ListFromStorageCallback(session)
            );
        }
    }

    private static class ListFromStorageCallback extends AbstractDocumentsProxySessionCallback<List<? extends Document>> {
        public ListFromStorageCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final List<? extends Document> documents) {
            try {
                JsonType jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter jw = jsonType.create(sbw)) {
                    jw.startObject();
                    jw.key("more");
                    jw.value(false);
                    jw.key("documents");
                    jw.value(documents);
                    jw.endObject();
                }

                session.response(
                    HttpStatus.SC_OK,
                    new StringEntity(sbw.toString(), ContentType.APPLICATION_JSON));
            } catch (IOException | BadRequestException ioe) {
                failed(ioe);
            }
        }
    }
}
