package ru.yandex.passport;

import java.io.IOException;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.client.tvm2.UserAuthResult;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServerException;
import ru.yandex.passport.document.DocumentSourceService;

public abstract class DocumentsProxyHandler implements HttpAsyncRequestHandler<HttpRequest> {
    private static final Set<DocumentSourceService> SERVICES_WITHOUT_USER_TICKET = Set.of(
            DocumentSourceService.MARKET,
            DocumentSourceService.TEST
    );

    protected final DocumentsProxy proxy;
    protected final boolean checkUserTicket;

    protected DocumentsProxyHandler(DocumentsProxy proxy, boolean checkUserTicket) {
        this.proxy = proxy;
        this.checkUserTicket = checkUserTicket;
    }

    protected DocumentsProxyHandler(DocumentsProxy proxy) {
        this(proxy, true);
    }

    @Override
    public void handle(
            final HttpRequest request,
            final HttpAsyncExchange exchange,
            final HttpContext context)
            throws HttpException, IOException
    {
        BasicProxySession session;
        try {
            session = new BasicProxySession(proxy, exchange, context);
        } catch (ServerException se) {
            ErrorFormattingCallback.writeErrorResponse(exchange, se.statusCode(), se.getMessage());
            return;
        } catch (Exception he) {
            ErrorFormattingCallback.writeErrorResponse(exchange, HttpStatus.SC_INTERNAL_SERVER_ERROR, he.getMessage());
            return;
        }
        try {
            DocumentSourceService service = RequestServiceResolver.resolve(session);
            if (checkUserTicket && !SERVICES_WITHOUT_USER_TICKET.contains(service))
            {
                UserAuthResult authResult = proxy.checkTvmUserAuthorization(request);
                if (authResult.errorDescription() != null) {
                    ErrorFormattingCallback.writeErrorResponse(exchange, HttpStatus.SC_FORBIDDEN, authResult.errorDescription());
                    return;
                }

                if (authResult.ticket().getDefaultUid() == 0L) {
                    // TODO we need to become sure about this behaviour (copied from SessionContext)
                    throw new BadRequestException("Default uid 0, multiauth is not supported here");
                }
                if (session.params().getLong("user_id", null) != authResult.ticket().getDefaultUid()) {
                    // TODO check all uids or not?
                    throw new BadRequestException("Ticket from another user");
                }
            }
            handle(session, service);
        } catch (Exception e) {
            ErrorFormattingCallback.writeErrorResponse(session, e);
        }
    }

    @Override
    public BasicAsyncRequestConsumer processRequest(
            final HttpRequest request,
            final HttpContext context) {
        return new BasicAsyncRequestConsumer();
    }

    public abstract void handle(
            final ProxySession session,
            final DocumentSourceService service)
            throws HttpException, IOException;
}
