package ru.yandex.passport;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.logging.Level;

import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.ServerException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;

public interface ErrorFormattingCallback<T> extends FutureCallback<T> {
    ProxySession session();

    @Override
    default void cancelled() {
        session().logger().warning(
            "Request cancelled: " + session().listener().details());
    }

    @Override
    default void failed(final Exception e) {
        int code = HttpStatus.SC_INTERNAL_SERVER_ERROR;
        if (e instanceof ServerException) {
            code = ((ServerException) e).statusCode();
        }
        String details = session().listener().details();
        session().logger().log(
            Level.WARNING,
            "Request failed: " + details + " because of",
            e);
        error(e, code);
    }

    default void error(final Exception e, final int code) {
        error(e.getMessage(), code);
    }

    default void error(final String message, final int code) {
        try {
            writeErrorResponse(session(), code, message);
        } catch (IOException ioe) {
            session().logger().log(Level.WARNING, "Failed to write response", ioe);
        }
    }
    static void writeErrorResponse(
        final ProxySession session,
        final Exception e)
        throws IOException
    {
        int code = HttpStatus.SC_INTERNAL_SERVER_ERROR;
        if (e instanceof ServerException) {
            code = ((ServerException) e).statusCode();
        }
        String details = session.listener().details();
        session.logger().log(
            Level.WARNING,
            "Request failed: " + details + " because of",
            e);
        writeErrorResponse(session, code, e.getMessage());
    }

    static void writeErrorResponse(
        final ProxySession session,
        final int status,
        final String error)
        throws IOException
    {
        session.response(
            status,
            generateMessage(error, session.acceptedCharset()));
    }

    static NStringEntity generateMessage(final String error, final Charset charset) throws IOException {
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
            writer.startObject();
            writer.key("status");
            writer.value("error");
            writer.key("error");
            writer.value(error);
            writer.endObject();
        }

        return new NStringEntity(
            sbw.toString(),
            ContentType.APPLICATION_JSON
                .withCharset(charset));
    }


    static void writeErrorResponse(
        final HttpAsyncExchange exchange,
        final int status,
        final String message)
        throws IOException
    {
        exchange.getResponse().setStatusCode(status);
        exchange.getResponse().setEntity(generateMessage(message, StandardCharsets.UTF_8));
        exchange.submitResponse();
    }
}
