package ru.yandex.passport;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.security.spec.AlgorithmParameterSpec;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.logging.Logger;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.passport.config.ImmutableAvatarConfig;
import ru.yandex.passport.document.BaseDocumentBuilder;
import ru.yandex.passport.document.Document;
import ru.yandex.passport.document.DocumentSourceService;
import ru.yandex.passport.document.DocumentType;
import ru.yandex.passport.document.ImageBuilder;
import ru.yandex.passport.document.ServiceEntry;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;

public class LuceneDocumentsStorage extends AbstractDocumentsStorage {
    private static final long FAILOVER_DELAY = 30;
    private static final boolean LOCALITY_SHUFFLE = true;

    private final AsyncClient client;
    private final HttpHost modifyHost;
    private final String queueName;
    private final DocumentsProxy proxy;
    private final ImmutableAvatarConfig avatarConfig;

    public LuceneDocumentsStorage(
            final DocumentsProxy proxy,
            final Key aesKey,
            final AlgorithmParameterSpec aesIv)
    {
        super(aesKey, aesIv);

        this.proxy = proxy;
        client = proxy.searchClient();
        avatarConfig = proxy.config().avatar();
        this.queueName = System.getProperty("QUEUE_NAME");
        this.modifyHost = proxy.producerHost();
        proxy.logger().info("Current service name " + queueName);
    }

    @Override
    public void get(
        final ProxySession session,
        final Long userId,
        final String documentId,
        final String originalId,
        final FutureCallback<BaseDocumentBuilder> callback)
    {
        SearchContext context = new SearchContext(session, userId);
        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", userId);
        try {
            qc.append("get", "*");
            if (documentId != null) {
                qc.append("text", "id_p:" + documentId);
            } else if (originalId != null) {
                qc.append("text", "original_id:" + originalId);
                qc.append("sort", "__queue_id");
                qc.append("group", "original_id");
                qc.append("collector", "pruning");
            } else {
                throw new BadRequestException("Both id and original id is null");
            }
        } catch (BadRequestException bre) {
            callback.failed(bre);
            return;
        }

        proxy.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            FAILOVER_DELAY,
            LOCALITY_SHUFFLE,
            JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING_OK,
            session.listener().createContextGeneratorFor(context.client()),
            new SingleDocumentFilterCallback(callback));
    }

    @Override
    public void list(
        final ProxySession session,
        final Long userId,
        final DocumentType documentType,
        final DocumentSourceService service,
        final FutureCallback<List<? extends Document>> callback)
    {
        SearchContext context = new SearchContext(session, userId);
        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", userId);
        try {
            qc.append("get", "doc_data,key_version,deleted,__queue_id");

            StringBuilder textBuilder = new StringBuilder();
            textBuilder.append("doc_type_p:");
            textBuilder.append(documentType == null ? "*" : documentType.name());
            textBuilder.append(" AND service_p:");
            textBuilder.append(service == null ? "*" : service.value());
            textBuilder.append(" AND deleted_p:false");
            qc.append("text", textBuilder.toString());

            qc.append("sort", "__queue_id");
            qc.append("group", "original_id");
            qc.append("collector", "pruning");

            qc.append("length", "100");
        } catch (BadRequestException bre) {
            callback.failed(bre);
            return;
        }


        proxy.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            FAILOVER_DELAY,
            LOCALITY_SHUFFLE,
            JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING_OK,
            session.listener().createContextGeneratorFor(context.client()),
            new DocumentListFilterCallback(callback));
    }

    @Override
    public void update(
        final ProxySession session,
        final Long userId,
        final BaseDocumentBuilder document,
        final FutureCallback<? super Document> callback)
    {
        put(session, userId, document, callback);
    }

    @Override
    public void put(
        final ProxySession session,
        final Long userId,
        final BaseDocumentBuilder document,
        final FutureCallback<? super Document> callback)
    {
        QueryConstructor qc = new QueryConstructor("/add?");
        if (document.id() == null && document.originalId() == null) {
            document.id(UUID.randomUUID().toString());
            document.createTime(OffsetDateTime.now());
            document.originalId(document.id());
        } else {
            document.originalId(document.originalId() != null ? document.originalId() : document.id());
            document.id(UUID.randomUUID().toString());
        }

        document.modificationTime(OffsetDateTime.now());
        qc.append("prefix", userId);
        try {
            qc.append("service", queueName);
            qc.append("wait", "true");
            qc.append("get", "*");
            qc.append("text", "doc_type:*");
            qc.append("length", "100");
        } catch (BadRequestException bre) {
            callback.failed(bre);
            return;
        }

        for (ImageBuilder image : document.images()) {
            image.originalUrl(null);
            image.previewUrl(null);
        }
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
            writer.startObject();
            writer.key("prefix");
            writer.value(userId);
            writer.key("docs");
            writer.startArray();
            writer.startObject();
            writer.key("id");
            writer.value(document.id());
            writer.key("doc_type");
            writer.value(document.docType().name());
            writer.key("doc_user_id");
            writer.value(userId);
            writer.key("service");
            writer.value(document.service().value());
            writer.key("original_id");
            writer.value(document.originalId());
            writer.key("deleted");
            writer.value(document.deleted());
            writer.key("key_version");
            writer.value(MANUAL_ENCRYPT_KEY_VERSION);
            writer.key("doc_data");
            writer.value(encrypt(JsonType.NORMAL.toString(document)));
            writer.endObject();
            writer.endArray();
            writer.endObject();
        } catch (IOException ioe) {
            callback.failed(ioe);
            return;
        }

        document.generateRetrieveUrls(avatarConfig);
        ProducerClient client = proxy.producerClient().adjust(session.context());
        client.execute(
            modifyHost,
            new BasicAsyncRequestProducerGenerator(qc.toString(), sbw.toString(), StandardCharsets.UTF_8),
            EmptyAsyncConsumerFactory.ANY_GOOD,
            session.listener().createContextGeneratorFor(client),
            new SaveCallback(callback, document));
    }

    private class SaveCallback extends AbstractFilterFutureCallback<Object, Document> {
        private final BaseDocumentBuilder document;

        public SaveCallback(final FutureCallback<? super Document> callback, final BaseDocumentBuilder document) {
            super(callback);
            this.document = document;
        }

        @Override
        public void completed(final Object o) {
            document.generateRetrieveUrls(avatarConfig);
            callback.completed(document);
        }
    }

    @Override
    public void delete(
        final ProxySession session,
        final Long userId,
        final String documentId,
        final String originalId,
        final FutureCallback<Object> callback)
    {
        if (documentId != null) {
            QueryConstructor qc = new QueryConstructor("/search?");
            qc.append("prefix", userId);
            try {
                qc.append("get", "original_id");
                qc.append("service", queueName);
                qc.append("text", "id_p:" + documentId);
            } catch (BadRequestException bre) {
                callback.failed(bre);
                return;
            }

            SearchContext context = new SearchContext(session, userId);
            proxy.sequentialRequest(
                    session,
                    context,
                    new BasicAsyncRequestProducerGenerator(qc.toString()),
                    FAILOVER_DELAY,
                    LOCALITY_SHUFFLE,
                    JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING_OK,
                    session.listener().createContextGeneratorFor(context.client()),
                    new DocumentDeleteCallback(callback, userId, session)
            );
        } else if (originalId != null) {
            QueryConstructor qc = new QueryConstructor("/update?");
            try {
                qc.append("prefix", userId);
                qc.append("service", queueName);
            } catch (BadRequestException e) {
                callback.failed(e);
                return;
            }

            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                writer.startObject();
                writer.key("prefix");
                writer.value(userId);
                writer.key("query");
                writer.value("original_id_p:" + originalId);
                writer.key("docs");
                writer.startArray();
                writer.startObject();
                writer.key("deleted");
                writer.value(true);
                writer.endObject();
                writer.endArray();
                writer.endObject();
            } catch (IOException ioe) {
                callback.failed(ioe);
            }

            ProducerClient client = proxy.producerClient().adjust(session.context());
            client.execute(
                    modifyHost,
                    new BasicAsyncRequestProducerGenerator(qc.toString(), sbw.toString(), StandardCharsets.UTF_8),
                    EmptyAsyncConsumerFactory.ANY_GOOD,
                    callback
            );
        } else {
            callback.failed(new BadRequestException("Both id and original id is null"));
        }
    }

    @Override
    public void getServices(
        final ProxySession session,
        final Long userId,
        final DocumentType documentType,
        final FutureCallback<List<ServiceEntry>> callback)
    {
        SearchContext context = new SearchContext(session, userId);
        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", userId);
        try {
            qc.append("get", "service,doc_type");
            if (documentType != null) {
                qc.append("text", "doc_type_p:" + documentType.name());
            } else {
                qc.append("text", "doc_type_p:*");
            }
            qc.append("group", "multi(service,doc_type)");
            qc.append("merge_func", "none");
        } catch (BadRequestException bre) {
            callback.failed(bre);
            return;
        }

        proxy.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            FAILOVER_DELAY,
            LOCALITY_SHUFFLE,
            JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING_OK,
            session.listener().createContextGeneratorFor(context.client()),
            new DocumentServicesFilterCallback(callback));
    }

    private class SingleDocumentFilterCallback extends AbstractFilterFutureCallback<JsonObject, BaseDocumentBuilder> {
        public SingleDocumentFilterCallback(final FutureCallback<? super BaseDocumentBuilder> callback) {
            super(callback);
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.asMap().getList("hitsArray");
                if (hits.size() == 0) {
                    callback.failed(new NotFoundException("Document not found"));
                    return;
                }

                String data = getDocDataDecrypted(hits.get(0));
                BaseDocumentBuilder builder = Document.parse(TypesafeValueContentHandler.parse(data).asMap());
                builder.generateRetrieveUrls(avatarConfig);
                builder.deleted(hits.get(0).asMap().getBoolean("deleted", false));
                builder.version(hits.get(0).asMap().getLong("__queue_id", 0L));
                callback.completed(builder);
            } catch (JsonException | IOException e) {
                callback.failed(e);
            }
        }
    }

    private class DocumentListFilterCallback
            extends AbstractFilterFutureCallback<JsonObject, List<? extends Document>> {
        public DocumentListFilterCallback(final FutureCallback<? super List<? extends Document>> callback) {
            super(callback);
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.asMap().getList("hitsArray");
                if (hits.size() == 0) {
                    callback.completed(Collections.emptyList());
                    //callback.failed(new NotFoundException("Document not found"));
                    return;
                }

                List<Document> documents = new ArrayList<>(hits.size());
                for (JsonObject hit : hits) {
                    String data = getDocDataDecrypted(hit);
                    BaseDocumentBuilder builder = Document.parse(TypesafeValueContentHandler.parse(data).asMap());
                    builder.generateRetrieveUrls(avatarConfig);
                    builder.deleted(hit.asMap().getBoolean("deleted", false));
                    builder.version(hit.asMap().getLong("__queue_id", 0L));
                    documents.add(builder);
                }

                callback.completed(documents);
            } catch (JsonException | IOException e) {
                callback.failed(e);
            }
        }
    }

    private static class DocumentServicesFilterCallback
            extends AbstractFilterFutureCallback<JsonObject, List<ServiceEntry>> {
        public DocumentServicesFilterCallback(final FutureCallback<List<ServiceEntry>> callback) {
            super(callback);
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.asMap().getList("hitsArray");
                if (hits.size() == 0) {
                    callback.completed(List.of());
                    return;
                }

                List<ServiceEntry> entries = new ArrayList<>(hits.size());
                for (JsonObject hit : hits) {
                    DocumentSourceService service = DocumentSourceService.of(hit.asMap().getString("service"));
                    DocumentType documentType = hit.asMap().getEnum(DocumentType.class, "doc_type");
                    entries.add(new ServiceEntry(
                            service,
                            documentType
                    ));
                }

                callback.completed(entries);
            } catch (JsonException je) {
                callback.failed(je);
            }
        }
    }

    private class SearchContext implements UniversalSearchProxyRequestContext {
        private final User user;
        private final ProxySession session;
        private final AsyncClient client;

        public SearchContext(final ProxySession session, final long userId) {
            this.session = session;
            this.user = new User(queueName, new LongPrefix(userId));
            this.client = LuceneDocumentsStorage.this.client.adjust(session.context());
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public long lagTolerance() {
            return 0;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public Logger logger() {
            return session.logger();
        }
    }

    private class DocumentDeleteCallback extends AbstractFilterFutureCallback<JsonObject, Object> {
        private final long userId;
        private final ProxySession session;

        protected DocumentDeleteCallback(FutureCallback<? super Object> callback, long userId, ProxySession session) {
            super(callback);
            this.userId = userId;
            this.session = session;
        }

        @Override
        public void completed(JsonObject result) {
            try {
                JsonList hits = result.asMap().getList("hitsArray");
                if (hits.size() == 0) {
                    callback.failed(new NotFoundException("Document not found"));
                    return;
                }

                String originalId = hits.get(0).asMap().getString("original_id");

                QueryConstructor qc = new QueryConstructor("/update?");
                qc.append("prefix", userId);
                qc.append("service", queueName);

                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                    writer.startObject();
                    writer.key("prefix");
                    writer.value(userId);
                    writer.key("query");
                    writer.value("original_id_p:" + originalId);
                    writer.key("docs");
                    writer.startArray();
                    writer.startObject();
                    writer.key("deleted");
                    writer.value(true);
                    writer.endObject();
                    writer.endArray();
                    writer.endObject();
                } catch (IOException ioe) {
                    callback.failed(ioe);
                }

                ProducerClient client = proxy.producerClient().adjust(session.context());
                client.execute(
                        modifyHost,
                        new BasicAsyncRequestProducerGenerator(qc.toString(), sbw.toString(), StandardCharsets.UTF_8),
                        EmptyAsyncConsumerFactory.ANY_GOOD,
                        callback);
            } catch (JsonException | BadRequestException e) {
                callback.failed(e);
            }
        }
    }
}
