package ru.yandex.passport.document;

import java.io.IOException;
import java.time.LocalDate;
import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.passport.config.ImmutableAvatarConfig;

public class BaseDocumentBuilder implements Document {
    private String id;
    private String docNumber;
    private long version;
    private String originalId;
    private String verificationData;
    private VerificationStatus verificationStatus;
    private String issuedForUser;
    private Boolean selfDoc;
    private Boolean defaultDoc;
    private Boolean draft;
    private String name;
    private List<ImageBuilder> images;
    private OffsetDateTime createTime;
    private OffsetDateTime modificationTime;
    private OffsetDateTime lastUsedTime;
    private DocumentType docType;
    private DocumentSourceService service;
    private boolean deleted;

    public BaseDocumentBuilder docType(final DocumentType docType) {
        this.docType = docType;
        return this;
    }

    @Override
    public String id() {
        return id;
    }

    public BaseDocumentBuilder id(final String id) {
        this.id = id;
        return this;
    }

    public String docNumber() {
        return docNumber;
    }

    public BaseDocumentBuilder docNumber(final String docNumber) {
        this.docNumber = docNumber;
        return this;
    }

    public long version() {
        return version;
    }

    public BaseDocumentBuilder version(final long version) {
        this.version = version;
        return this;
    }

    public String verificationData() {
        return verificationData;
    }

    public BaseDocumentBuilder verificationData(final String verificationData) {
        this.verificationData = verificationData;
        return this;
    }

    public VerificationStatus verificationStatus() {
        return verificationStatus;
    }

    public BaseDocumentBuilder verificationStatus(final VerificationStatus verificationStatus) {
        this.verificationStatus = verificationStatus;
        return this;
    }

    public String issuedForUser() {
        return issuedForUser;
    }

    public BaseDocumentBuilder issuedForUser(final String issuedForUser) {
        this.issuedForUser = issuedForUser;
        return this;
    }

    public Boolean selfDoc() {
        return selfDoc;
    }

    public BaseDocumentBuilder selfDoc(final Boolean selfDoc) {
        this.selfDoc = selfDoc;
        return this;
    }

    public Boolean defaultDoc() {
        return defaultDoc;
    }

    public BaseDocumentBuilder defaultDoc(final Boolean defaultDoc) {
        this.defaultDoc = defaultDoc;
        return this;
    }

    public OffsetDateTime createTime() {
        return createTime;
    }

    public BaseDocumentBuilder createTime(final OffsetDateTime createTime) {
        this.createTime = createTime;
        return this;
    }

    public OffsetDateTime modificationTime() {
        return modificationTime;
    }

    public BaseDocumentBuilder modificationTime(final OffsetDateTime modificationTime) {
        this.modificationTime = modificationTime;
        return this;
    }

    public OffsetDateTime lastUsedTime() {
        return lastUsedTime;
    }

    public BaseDocumentBuilder lastUsedTime(final OffsetDateTime lastUsedTime) {
        this.lastUsedTime = lastUsedTime;
        return this;
    }

    public Boolean draft() {
        return draft;
    }

    public BaseDocumentBuilder draft(final Boolean draft) {
        this.draft = draft;
        return this;
    }

    public String name() {
        return name;
    }

    public BaseDocumentBuilder name(final String name) {
        this.name = name;
        return this;
    }

    public DocumentSourceService service() {
        return service;
    }

    public BaseDocumentBuilder service(DocumentSourceService service) {
        this.service = service;
        return this;
    }

    public List<ImageBuilder> images() {
        return images != null ? images : List.of();
    }

    public BaseDocumentBuilder images(List<ImageBuilder> images) {
        this.images = images;
        return this;
    }

    @Override
    public DocumentType docType() {
        return docType;
    }

    @Override
    public void writeFields(final JsonWriterBase writer) throws IOException {
        writer.key("doc_type");
        writer.value(docType().name().toLowerCase(Locale.ENGLISH));
        writer.key("id");
        writer.value(id());
        writer.key("create_time");
        writer.value(createTime());
        writer.key("modification_time");
        writer.value(modificationTime());
        writer.key("last_used_time");
        writer.value(lastUsedTime());
        writer.key("version");
        writer.value(version());
        writer.key("original_id");
        writer.value(originalId());
        writer.key("verification_data");
        writer.value(verificationData());
        writer.key("verification_status");
        writer.value(verificationStatus());
        writer.key("issued_for_user");
        writer.value(issuedForUser());
        writer.key("self_doc");
        writer.value(selfDoc());
        writer.key("default");
        writer.value(defaultDoc());
        writer.key("doc_number");
        writer.value(docNumber());
        writer.key("images");
        writer.startArray();
        for (ImageBuilder image: images()) {
            writer.value(image);
        }
        writer.endArray();

        // TODO eshemchik: delete on front readiness
        if (!images().isEmpty()) {
            ImageBuilder firstImage = images().get(0);
            writer.key("image_id");
            writer.value(firstImage.imageId());
            writer.key("original_uri");
            writer.value(firstImage.originalUri());
            writer.key("original_width");
            writer.value(firstImage.originalWidth());
            writer.key("original_height");
            writer.value(firstImage.originalHeight());
            writer.key("preview_uri");
            writer.value(firstImage.previewUri());
            if (firstImage.previewUrl() != null) {
                writer.key("preview_url");
                writer.value(firstImage.previewUrl());
            }
            if (firstImage.originalUrl() != null) {
                writer.key("original_url");
                writer.value(firstImage.originalUrl());
            }
            writer.key("preview_width");
            writer.value(firstImage.previewWidth());
            writer.key("preview_height");
            writer.value(firstImage.previewHeight());
        } else {
            writer.key("image_id");
            writer.value(JsonNull.INSTANCE);
            writer.key("original_uri");
            writer.value(JsonNull.INSTANCE);
            writer.key("original_width");
            writer.value(JsonNull.INSTANCE);
            writer.key("original_height");
            writer.value(JsonNull.INSTANCE);
            writer.key("preview_uri");
            writer.value(JsonNull.INSTANCE);
            writer.key("preview_url");
            writer.value(JsonNull.INSTANCE);
            writer.key("original_url");
            writer.value(JsonNull.INSTANCE);
            writer.key("preview_width");
            writer.value(JsonNull.INSTANCE);
            writer.key("preview_height");
            writer.value(JsonNull.INSTANCE);
        }

        writer.key("draft");
        writer.value(draft());
        writer.key("service");
        // TODO eshemchik: delete default after full migration
        writer.value(
                service != null
                        ? service.value()
                        : DocumentSourceService.PASSPORT.value()
        );
        writer.key("deleted");
        writer.value(deleted());
    }

    public BaseDocumentBuilder parse(JsonMap map) throws JsonException {
        this.docType(map.getEnum(DocumentType.class, "doc_type"));
        this.id(map.getString("id", null));
        this.originalId(map.getString("original_id", null));
        this.createTime(map.get("create_time", null, OffsetDateTime::parse));
        this.modificationTime(map.get("modification_time", null, OffsetDateTime::parse));
        this.lastUsedTime(map.get("last_used_time", null, OffsetDateTime::parse));
        this.version(map.getLong("version", 0L));
        this.verificationData(map.getString("verification_data", null));
        this.verificationStatus(map.getEnum(VerificationStatus.class, "verification_status", null));
        this.issuedForUser(map.getString("issued_for_user", null));
        this.selfDoc(map.getBoolean("self_doc", null));
        this.draft(map.getBoolean("draft", false));
        this.name(map.getString("name", null));
        this.defaultDoc(map.getBoolean("default", null));
        this.docNumber(map.getString("doc_number", null));
        this.deleted(map.getBoolean("deleted", false));
        // TODO eshemchik: delete after migration
        if (map.containsKey("images")) {
            JsonList imagesList = map.getList("images");
            List<ImageBuilder> images = new ArrayList<>(imagesList.size());
            for (JsonObject object : imagesList) {
                images.add(ImageBuilder.parse(object.asMap()));
            }
            this.images(images);
        } else if (map.containsKey("image_id") && map.get("image_id") != JsonNull.INSTANCE) {
            this.images(List.of(ImageBuilder.parse(map)));
        }
        this.draft(map.getBoolean("draft", null));
        if (map.containsKey("service")) {
            this.service(DocumentSourceService.of(map.getString("service", null)));
        }
        return this;
    }

    public void generateRetrieveUrls(ImmutableAvatarConfig avatarConfig) {
        for (ImageBuilder image : images()) {
            image.generateRetrieveUrls(avatarConfig);
        }
    }

    protected LocalDate parseLocalDate(String value) {
        return value == null ? null : LocalDate.parse(value, DateTimeFormatter.ISO_DATE);
    }

    public String originalId() {
        return originalId;
    }

    public BaseDocumentBuilder originalId(String originalId) {
        this.originalId = originalId;
        return this;
    }

    public boolean deleted() {
        return deleted;
    }

    public BaseDocumentBuilder deleted(boolean deleted) {
        this.deleted = deleted;
        return this;
    }
}
