package ru.yandex.passport;

import java.util.Arrays;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;

import ru.yandex.blackbox.BlackboxSessionUserinfo;
import ru.yandex.blackbox.BlackboxSessionidRequest;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ByteArrayEntityFactory;
import ru.yandex.http.util.ByteArrayProcessableWithContentType;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.ByteArrayProcessableWithContentTypeAsyncConsumerFactory;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.uri.QueryConstructor;

public class GetImageHandler implements ProxyRequestHandler {
    private final ImagesProxy proxy;

    public GetImageHandler(ImagesProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void handle(ProxySession session) {
        ImageProxyRequestContext context;
        BlackboxSessionidRequest bbRequest;
        try {
            context = new ImageProxyRequestContext(session);

            bbRequest =
                    new BlackboxSessionidRequest(
                            parseSessionId(session),
                            session.headers().getString("Host")
                    );
            bbRequest.ip(session.headers().getString(YandexHeaders.X_FORWARDED_FOR_Y));
            bbRequest.addHeader(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    proxy.blackboxTvm2Ticket()
            );
        } catch (HttpException e) {
            session.logger().log(Level.WARNING, "Error ", e);
            session.response(HttpStatus.SC_INTERNAL_SERVER_ERROR);
            return;
        }

        proxy.blackboxClient().sessionid(
                bbRequest,
                session.listener().adjustContextGenerator(proxy.blackboxClient().httpClientContextGenerator()),
                new BlackboxCallback(context)
        );
    }

    private String parseSessionId(ProxySession session) throws BadRequestException {
        return Arrays.stream(session.headers().getString("Cookie").split("; "))
                .filter(s -> s.startsWith("Session_id="))
                .map(s -> s.substring("Session_id=".length()))
                .findAny()
                .orElseThrow(() -> new BadRequestException("SessionId is not present"));
    }

    private class BlackboxCallback extends ImagesProxyCallback<BlackboxSessionUserinfo> {
        private BlackboxCallback(ImageProxyRequestContext context) {
            super(context);
        }

        @Override
        public void completed(BlackboxSessionUserinfo blackboxSessionUserinfo) {
            QueryConstructor qc = new QueryConstructor("/document/image/check?");
            try {
                qc.append("user_id", blackboxSessionUserinfo.uid());
                qc.append("doc_id", context.documentId());
                qc.append("path", context.path());
            } catch (BadRequestException e) {
                failed(e);
            }
            BasicAsyncRequestProducerGenerator generator = new BasicAsyncRequestProducerGenerator(qc.toString());
            AsyncClient client = proxy.documentsProxyClient().adjust(session().context());
            client.execute(
                    proxy.config().documentsProxy().host(),
                    generator,
                    EmptyAsyncConsumerFactory.ANY_GOOD,
                    session().listener().adjustContextGenerator(client.httpClientContextGenerator()),
                    new DocumentsProxyCallback(context)
            );
        }
    }

    private final class DocumentsProxyCallback extends ImagesProxyCallback<Object> {
        public DocumentsProxyCallback(ImageProxyRequestContext context) {
            super(context);
        }

        @Override
        public void completed(Object o) {
            QueryConstructor qc = new QueryConstructor("/get-id_doc/" + context.path());
            BasicAsyncRequestProducerGenerator generator = new BasicAsyncRequestProducerGenerator(qc.toString());
            generator.copyHeader(session().request(), HttpHeaders.CONTENT_TYPE);

            AsyncClient client = proxy.avatarsClient().adjust(session().context());
            client.execute(
                    proxy.config().avatar().host(),
                    generator,
                    ByteArrayProcessableWithContentTypeAsyncConsumerFactory.OK,
                    session().listener().adjustContextGenerator(client.httpClientContextGenerator()),
                    new AvatarsCallback(context)
            );
        }

        @Override
        public void failed(Exception e) {
            session().response(HttpStatus.SC_FORBIDDEN);
            session().logger().log(Level.INFO, "Access forbidden ", e);
        }
    }


    private static class AvatarsCallback extends ImagesProxyCallback<ByteArrayProcessableWithContentType> {
        private AvatarsCallback(ImageProxyRequestContext context) {
            super(context);
        }

        @Override
        public void completed(ByteArrayProcessableWithContentType data) {
            session().getResponse().addHeader(data.contentTypeHeader());
            session().getResponse().addHeader(data.contentEncodingHeader());
            session().response(HttpStatus.SC_OK, data.data().processWith(ByteArrayEntityFactory.INSTANCE));
        }
    }
}
