package ru.yandex.passport.search;

import java.io.File;
import java.io.IOException;
import java.nio.charset.CharacterCodingException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.uri.PctEncoder;
import ru.yandex.parser.uri.PctEncodingRule;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.passport.search.config.ImmutableMultisearchConfig;

public class PassportMultidataSearchHandler implements ProxyRequestHandler {
    public static final String MAIL_BASE_SEARCH_URL =
        "https://mail.yandex.ru/?#search?request={request}";
    public static final String DISK_BASE_SEARCH_URL =
        "https://disk.yandex.ru/client/search?&scopeSearch=%2Fdisk&querySearch={request}";

//    public static final HttpHost MAIL_SEARCH_HOST
//        = new HttpHost("new-msearch-proxy.mail.yandex.net", 8051);
//    public static final HttpHost PASSPORT_HOST
//        = new HttpHost("localhost", 91);
//    private static final HttpHost DISK_SEARCH_HOST
//        = new HttpHost("disk-proxy.pers.yandex.net", 443, "https");
//    private static final HttpHost DISK_SEARCH_HOST
//        = new HttpHost("man1-7315-disksearch-yadisk-proxy-19502.gencfg-c.yandex.net", 19502, "http");
//    private static final HttpHost MESSENGER_SEARCH_HOST
//        = new HttpHost("messenger-search-proxy.pers.yandex.net", 443, "https");
    private final HttpProxy<?> server;
    final AsyncClient mailClient;
    final AsyncClient diskClient;
    //final AsyncClient passportClient;
    final AsyncClient messengerClient;

    private final ImmutableMultisearchConfig config;

    private final List<SearchResultProvider> providers;

    public PassportMultidataSearchHandler(
        final HttpProxy<?> server,
        final ImmutableMultisearchConfig config)
        throws IOException
    {
        this.config = config;
        ImmutableHttpTargetConfig clientConfig;
        try {
            HttpTargetConfigBuilder builder = new HttpTargetConfigBuilder();
            builder.connections(5);
            builder.timeout(5000);
            clientConfig = builder.build();
        } catch (ConfigException ce) {
            throw new IOException(ce);
        }

        providers = Arrays.asList(
            new PassportPortal(server, config.passport()),
            new MailSearchResultProvider(),
            new DiskSearchResultProvider());

        mailClient = server.client("MailSearch", config.mail());
        diskClient = server.client("DiskSearch", config.disk());
        //passportClient = server.client("Passport", config.passport());
        messengerClient = server.client("Messenger", clientConfig);

        this.server = server;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        PassportMultidataSearchContext context = new PassportMultidataSearchContext(session);
        ResultPrinter printer = new ResultPrinter(context);
        MultiFutureCallback<Map.Entry<SearchSource, List<SerpItem>>> mfcb
            = new MultiFutureCallback<>(printer);
        for (SearchResultProvider provider: providers) {
            provider.execute(context, mfcb.newCallback());
        }
        mfcb.done();
    }

    private class MailSearchResultProvider implements SearchResultProvider {
        @Override
        public void execute(
            final PassportMultidataSearchContext context,
            final FutureCallback<Map.Entry<SearchSource, List<SerpItem>>> callback)
            throws BadRequestException,IOException
        {
            QueryConstructor qc = new QueryConstructor("/api/async/mail/search?");
            qc.append("source", "passport_id");
            qc.append("request", context.request());
            qc.append("count", context.length());
            qc.append("first", 0);
            qc.append("uid", context.uid());
            qc.append("mdb", "pg");
            qc.copyIfPresent(context.session().params(), "remote_ip");

            AsyncClient client =
                mailClient.adjust(context.session().context());
            client.execute(
                Collections.singletonList(server.httpHost()),
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.session().listener().createContextGeneratorFor(client),
                new MailSearchSourceCallback(context, callback));
        }
    }

    private class DiskSearchResultProvider implements SearchResultProvider {
        @Override
        public void execute(
            final PassportMultidataSearchContext context,
            final FutureCallback<Map.Entry<SearchSource, List<SerpItem>>> callback)
            throws BadRequestException,IOException
        {
            //curl 'localhost:19502/?service=disk&kps=1044774674&text=1430&p=16&nudoc=10&fast-moved&format=json&only=id,key,name,ctime,etime,photoslice_time&aux_folder=disk&aux_folder=photounlim&hr'
            QueryConstructor qc = new QueryConstructor("/?");
            qc.append("source", "passport_id");
            qc.append("text", context.request());
            qc.append("service", "disk");
            qc.append("numdoc", context.length());
            //qc.append("first", 0);
            qc.append("kps", context.uid());
            qc.append("only", "id,key,name,ctime,etime,photoslice_time,preview_stid,mimetype");
            qc.append("format", "json");
            qc.append("fast-moved", "true");
            qc.append("aux_folder", "disk");
            qc.append("aux_folder", "photounlim");
            qc.copyIfPresent(context.session().params(), "remote_ip");

            AsyncClient client =
                diskClient.adjust(context.session().context());
            client.execute(
                Collections.singletonList(config.disk().host()),
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.session().listener().createContextGeneratorFor(client),
                new DiskSearchSourceCallback(context, callback));
        }
    }

//    private class PassportSearchResultProvider implements SearchResultProvider {
//        @Override
//        public void execute(
//            final PassportMultidataSearchContext context,
//            final FutureCallback<Map.Entry<SearchSource, List<SerpItem>>> callback)
//            throws BadRequestException,IOException
//        {
//            //curl 'localhost:19502/?service=disk&kps=1044774674&text=1430&p=16&nudoc=10&fast-moved&format=json&only=id,key,name,ctime,etime,photoslice_time&aux_folder=disk&aux_folder=photounlim&hr'
//            QueryConstructor qc = new QueryConstructor("/search?");
//            qc.append("source", "passport_id");
//            qc.append("text", SearchRequestText.parseSuggest(context.request(), Locale.ROOT).fieldsQuery("page_title", "link_title"));
//            //qc.append("service", "passport");
//            qc.append("length", context.length());
//            qc.append("get", "*");
//            AsyncClient client =
//                passportClient.adjust(context.session().context());
//            client.execute(
//                Collections.singletonList(PASSPORT_HOST),
//                new BasicAsyncRequestProducerGenerator(qc.toString()),
//                JsonAsyncTypesafeDomConsumerFactory.OK,
//                context.session().listener().createContextGeneratorFor(client),
//                new PassportSearchCallback(context, callback));
//        }
//    }

    private static class DiskSearchSourceCallback
        extends AbstractFilterFutureCallback<JsonObject, Map.Entry<SearchSource, List<SerpItem>>>
    {
        private final PassportMultidataSearchContext context;
        public DiskSearchSourceCallback(
            final PassportMultidataSearchContext context,
            final FutureCallback<? super Map.Entry<SearchSource, List<SerpItem>>> callback)
        {
            super(callback);

            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            try {
                JsonMap resultMap = resultObj.asMap();
                PctEncoder encoder = new PctEncoder(PctEncodingRule.QUERY);
                List<SerpItem> diskResult = new ArrayList<>(resultMap.getMap("response").getMap("found").getInt("all"));
                JsonList results = resultMap.getMap("response").getList("results");
                for (JsonObject result: results) {
                    for (JsonObject group: result.asMap().getList("groups")) {
                        for (JsonObject document: group.asMap().getList("documents")) {
                            diskResult.add(new DiskSerpItem(encoder, document.asMap()));
                        }
                    }
                }
                context.session().logger().info("Disk search found " + diskResult.size());

                callback.completed(
                    new AbstractMap.SimpleEntry<>(SearchSource.DISK, diskResult));
            } catch (JsonException | CharacterCodingException e) {
                failed(e);
            }
        }
    }

    private static class PassportSearchCallback
        extends AbstractFilterFutureCallback<JsonObject, Map.Entry<SearchSource, List<SerpItem>>>
    {
        private final PassportMultidataSearchContext context;
        public PassportSearchCallback(
            final PassportMultidataSearchContext context,
            final FutureCallback<? super Map.Entry<SearchSource, List<SerpItem>>> callback)
        {
            super(callback);

            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            try {
                JsonMap resultMap = resultObj.asMap();
                JsonList hits = resultMap.getList("hitsArray");
                List<SerpItem> psspResult = new ArrayList<>(hits.size());
                for (JsonObject hitObj: hits) {
                    JsonMap hit = hitObj.asMap();
                    psspResult.add(
                        new BasicSearchItem(
                            SearchSource.PASSPORT,
                            hit.getString("page_title"),
                            hit.getString("link_title"),
                            hit.getString("url"),
                            hit.getString("url"),
                            null,
                            0));
                }
                context.session().logger().info("Passport search found " + psspResult);

                callback.completed(
                    new AbstractMap.SimpleEntry<>(SearchSource.PASSPORT, psspResult));
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    private static class DiskSerpItem implements SerpItem {
        private final String id;
        private final String key;
        private final String name;
        private final String preview;
        private final String mimetype;
        private final String scope;
        private final String resourceUrl;
        private final String avatarUrl;
        private final long date;

        public DiskSerpItem(
            final PctEncoder encoder,
            final JsonMap item)
            throws JsonException, CharacterCodingException
        {
            JsonMap properties = item.getMap("properties");
            this.id = properties.getString("id");
            this.name = properties.getString("name");
            Long date = properties.getLong("ctime", null);
            date = properties.getLong("etime", date);
            if (date == null) {
                date = 0L;
            }
            this.date = date * 1000;
            this.key = properties.getString("key");
            this.scope = properties.getString("scope");
            this.preview = properties.getString("preview_stid", null);
            this.mimetype = properties.getString("mimetype", null);
            //System.out.println(JsonType.HUMAN_READABLE.toString(item));
            StringBuilder rurl = new StringBuilder();
            rurl.append("https://disk.yandex.ru/client");
            if (!"folder".equalsIgnoreCase(scope)) {
                String parent = new File(key).getParent();
                rurl.append(parent);
                //encoder.clear();
                //encoder.process(key.toCharArray());
                rurl.append("/?idApp=client&dialog=slider&idDialog=");
                rurl.append(key);
                //encoder.toStringBuilder(rurl);
            } else {
                rurl.append(key);
            }

            this.resourceUrl = rurl.toString();

            this.avatarUrl = null;
        }


        @Override
        public SearchSource source() {
            return SearchSource.DISK;
        }

        @Override
        public String id() {
            return id;
        }

        @Override
        public long date() {
            return date;
        }

        @Override
        public String name() {
            return name;
        }

        @Override
        public String resourceUrl() {
            return resourceUrl;
        }

        @Override
        public String avatarUrl() {
            return avatarUrl;
        }

        @Override
        public String snippet() {
            return "";
        }

        @Override
        public void writeExtraFields(final JsonWriterBase writer)
            throws IOException
        {
            writer.key("path");
            writer.value(key);
            writer.key("mimetype");
            writer.value(mimetype);
            writer.key("scope");
            writer.value(scope);
        }
    }

    private static class MailSearchSourceCallback
        extends AbstractFilterFutureCallback<JsonObject, Map.Entry<SearchSource, List<SerpItem>>>
    {
        private final PassportMultidataSearchContext context;
        public MailSearchSourceCallback(
            final PassportMultidataSearchContext context,
            final FutureCallback<? super Map.Entry<SearchSource, List<SerpItem>>> callback)
        {
            super(callback);

            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            try {
                JsonMap resultMap = resultObj.asMap();
                JsonList envelopes = resultMap.getList("envelopes");
                context.session().logger().info("Mail search found " + envelopes.size());
                List<SerpItem> result = new ArrayList<>(envelopes.size());
                for (JsonObject envelope: envelopes) {
                    result.add(new MailSerpItem(context.uid(), envelope.asMap()));
                }

                callback.completed(
                    new AbstractMap.SimpleEntry<>(SearchSource.MAIL, result));
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    private static class MailSerpItem implements SerpItem {
        private final String from;
        private final String mid;
        private final String firstline;
        private final String subject;
        private final String resourceUrl;
        private final long date;

        public MailSerpItem(final Long uid, final JsonMap envelope) throws JsonException {
            this.firstline = envelope.getString("firstline");
            this.mid = envelope.getString("mid");
            this.subject = envelope.getMap("subjectInfo").getString("subject");
            JsonList fromList = envelope.getList("from");
            JsonMap fromMap = fromList.get(0).asMap();
            this.from = fromMap.getString("local") + '@' + fromMap.getString("domain");
            this.date = envelope.getLong("receiveDate") * 1000;
            this.resourceUrl = "https://mail.yandex.ru/?uid=" + uid + "#message/" + mid;
        }

        public String from() {
            return from;
        }

        public String mid() {
            return mid;
        }

        public String firstline() {
            return firstline;
        }

        @Override
        public String name() {
            return subject;
        }

        @Override
        public long date() {
            return date;
        }

        @Override
        public SearchSource source() {
            return SearchSource.MAIL;
        }

        @Override
        public String id() {
            return mid;
        }

        @Override
        public String resourceUrl() {
            return resourceUrl;
        }

        @Override
        public String avatarUrl() {
            return "";
        }

        @Override
        public String snippet() {
            return subject;
        }

        @Override
        public void writeExtraFields(final JsonWriterBase writer)
            throws IOException
        {
            writer.key("from");
            writer.value(from);
            writer.key("firstline");
            writer.value(firstline);
        }
    }

    private static class BasicSearchItem implements SerpItem {
        private final SearchSource source;
        private final String name;
        private final String snippet;
        private final String id;
        private final String resourceUrl;
        private final String avatarUrl;
        private final long date;

        public BasicSearchItem(
            final SearchSource source,
            final String name,
            final String snippet,
            final String id,
            final String resourceUrl,
            final String avatarUrl,
            final long date)
        {
            this.source = source;
            this.name = name;
            this.snippet = snippet;
            this.id = id;
            this.resourceUrl = resourceUrl;
            this.avatarUrl = avatarUrl;
            this.date = date;
        }

        @Override
        public SearchSource source() {
            return source;
        }

        @Override
        public String id() {
            return id;
        }

        @Override
        public long date() {
            return date;
        }

        @Override
        public String name() {
            return name;
        }

        @Override
        public String resourceUrl() {
            return resourceUrl;
        }

        @Override
        public String avatarUrl() {
            return avatarUrl;
        }

        @Override
        public String snippet() {
            return snippet;
        }
    }

    private static class ResultPrinter
        extends AbstractProxySessionCallback<List<Map.Entry<SearchSource, List<SerpItem>>>>
    {
        private final PassportMultidataSearchContext context;
        private final JsonType jsonType;

        public ResultPrinter(
            final PassportMultidataSearchContext context)
            throws BadRequestException
        {
            super(context.session());
            this.context = context;
            this.jsonType = JsonTypeExtractor.NORMAL.extract(context.session().params());
        }

        @Override
        public void completed(final List<Map.Entry<SearchSource, List<SerpItem>>> result) {
            StringBuilderWriter sbw = new StringBuilderWriter();

            try (JsonWriterBase writer = jsonType.create(sbw)) {
                writer.startObject();
                writer.key("results");
                writer.startArray();
                for (Map.Entry<SearchSource, List<SerpItem>> entry: result) {
                    writer.startObject();
                    writer.key("source");
                    writer.value(entry.getKey());
                    writer.key("search_url");
                    writer.value(entry.getKey().searchUrl(context.uid(), context.request()));
                    writer.key("results");
                    writer.value(entry.getValue());
                    writer.endObject();
                }
                writer.endArray();
                writer.endObject();
            } catch (IOException ioe) {
                failed(ioe);
                return;
            }

            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON
                        .withCharset(context.session().acceptedCharset())));

        }
    }

}
