package ru.yandex.passport.search;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Files;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;

import org.apache.commons.collections4.trie.PatriciaTrie;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.client.CloseableHttpClient;

import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.passport.search.config.ImmutablePassportPortalConfig;

public class PassportPortal implements SearchResultProvider {
    private static final long BUNKER_UPDATE_PERIOD = TimeUnit.MINUTES.toMillis(5);
    private final ImmutablePassportPortalConfig config;

    private final PrefixedLogger logger;
    private final AtomicReference<PatriciaTrie<PassportItem>> trie;
    private final CloseableHttpClient bunkerClient;
    private final String bunkerUri;
    private final Timer timer;

    public PassportPortal(
        final HttpProxy<?> server,
        final ImmutablePassportPortalConfig config) throws IOException
    {
        this.config = config;
        PrefixedLogger logger = server.logger();
        if (config.staticFile() == null && config.bunker() == null) {
            throw new IOException("No static and no bunker specified");
        }

        if (config.staticFile() != null) {
            trie = new AtomicReference<>(loadFromFile(config.staticFile()));
            this.bunkerUri = null;
            this.timer = null;
            this.bunkerClient = null;
        } else {
            this.bunkerClient = ClientBuilder.createClient(config.bunker(), server.config().dnsConfig());
            if (config.cacheFile() != null) {
                Files.createDirectories(config.cacheFile().toPath());
            }

            QueryConstructor bunkerUri = new QueryConstructor("/v1/cat?");
            try {
                bunkerUri.append("node", config.bunker().uri().getPath());
            } catch (BadRequestException bre) {
                throw new IOException(bre);
            }
            this.bunkerUri = bunkerUri.toString();
            PatriciaTrie<PassportItem> trie;
            try {
                trie = loadBunker();
            } catch (Exception e) {
                logger.log(Level.WARNING, "Failed to load index from bunker", e);
                trie = loadFromFile(config.cacheFile());
            }
            this.timer = new Timer(true);
            this.timer.scheduleAtFixedRate(new BunkerUpdater(), BUNKER_UPDATE_PERIOD, BUNKER_UPDATE_PERIOD);
            this.trie = new AtomicReference<>(trie);
        }

        this.logger = logger;
        logger.info("Trie " + trie);
    }

    public PrefixedLogger logger() {
        return logger;
    }

    private PatriciaTrie<PassportItem> parseIndex(final JsonObject jsonObject) throws JsonException {
        PatriciaTrie<PassportItem> trie = new PatriciaTrie<>();
        for (JsonObject jo: jsonObject.asList()) {
            JsonMap map = jo.asMap();
            String name = map.getString("name");
            String id = map.getString("id");
            String resourceUrl = map.getString("resource_url");
            PassportItem item = new PassportItem(id, name, config.clickUrlTemplate() + resourceUrl);
            indexToTrie(name, item, trie);
        }

        return trie;
    }

    private PatriciaTrie<PassportItem> loadFromFile(final File file) throws IOException {
        try (FileReader reader = new FileReader(file)) {
            JsonList index = TypesafeValueContentHandler.parse(reader).asList();
            return parseIndex(index);
        } catch (JsonException je) {
            throw new IOException("Failed to init passport portal", je);
        }
    }

    private PatriciaTrie<PassportItem> loadBunker() throws Exception {
        HttpGet request = new HttpGet(bunkerUri);
        try (CloseableHttpResponse response =
                 bunkerClient.execute(
                     config.bunker().host(),
                     request))
        {
            if (HttpStatusPredicates.ANY_GOOD.test(response.getStatusLine().getStatusCode())) {
                return parseIndex(TypesafeValueContentHandler.parse(CharsetUtils.content(response.getEntity())));
            }

            throw new BadResponseException(request, response);
        }
    }

    @Override
    public void execute(
        final PassportMultidataSearchContext context,
        final FutureCallback<Map.Entry<SearchSource, List<SerpItem>>> callback)
        throws BadRequestException, IOException
    {
        String request = normalizeKey(context.request());
        SortedMap<String, PassportItem> resultMap = trie.get().prefixMap(request);
        Set<SerpItem> result = new LinkedHashSet<>();
        for (Map.Entry<String, PassportItem> entry: resultMap.entrySet()) {
            result.add(entry.getValue());
        }

        logger.info("Found for passport " + resultMap + " request: " + request);
        callback.completed(new AbstractMap.SimpleEntry<>(SearchSource.PASSPORT, new ArrayList<>(result)));
    }

    private String normalizeKey(final String key) {
        return key.toLowerCase(Locale.ROOT).replaceAll("ё", "е");
    }

    private void indexToTrie(final String key, final PassportItem value, final PatriciaTrie<PassportItem> trie) {
        String[] split = key.split("[^\\p{L}0-9']+");

        List<String> tokens =  new ArrayList<>();
        tokens.addAll(Arrays.asList(split));
        tokens.add(key);
        for (String token: tokens) {
            trie.put(normalizeKey(token), value);
        }
    }

    private static class PassportItem implements SerpItem {
        private final String id;
        private final String name;
        private final String resourceUrl;

        public PassportItem(final String id, final String name, final String resourceUrl) {
            this.id = id;
            this.name = name;
            this.resourceUrl = resourceUrl;
        }

        @Override
        public SearchSource source() {
            return SearchSource.PASSPORT;
        }

        @Override
        public String id() {
            return id;
        }

        @Override
        public long date() {
            return 0;
        }

        @Override
        public String name() {
            return name;
        }

        @Override
        public String resourceUrl() {
            return resourceUrl;
        }

        @Override
        public String avatarUrl() {
            return null;
        }

        @Override
        public String snippet() {
            return name;
        }
    }

    private class BunkerUpdater extends TimerTask {
        @Override
        public void run() {
            try {
                PatriciaTrie<PassportItem> trie = loadBunker();
                PassportPortal.this.trie.set(trie);
            } catch (Exception e) {
                logger.log(Level.WARNING, "Failed to load bunker", e);
            }
        }
    }
}
