package ru.yandex.passport.phone.ownership;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.passport.phone.ownership.edna.EdnaRequestUtils;
import ru.yandex.passport.phone.ownership.edna.ProcessImsiResponseCallback;
import ru.yandex.passport.phone.ownership.parse.PhoneParseResult;
import ru.yandex.passport.phone.ownership.parse.PhoneParseStatus;
import ru.yandex.passport.phone.ownership.parse.PhoneParser;

public class PhoneChangeStatusHandler implements ProxyRequestHandler {
    private final PhoneOwnershipProxy proxy;
    private final boolean skipActualEdna;

    public PhoneChangeStatusHandler(final PhoneOwnershipProxy proxy) {
        this.proxy = proxy;
        this.skipActualEdna =
            Boolean.parseBoolean(
                System.getProperty("SKIP_EDNA", Boolean.TRUE.toString()));
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        HttpRequest request = session.request();
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        HttpEntityEnclosingRequest postRequest = (HttpEntityEnclosingRequest) request;
        HttpEntity entity = postRequest.getEntity();
        PhoneChangeStatusPrinter callback = new PhoneChangeStatusPrinter(session);
        try {
            JsonMap map = TypesafeValueContentHandler.parse(
                CharsetUtils.content(entity)).asMap();
            String phoneStr = map.getString("phone");
            session.connection().setSessionInfo(
                "phoneStr",
                phoneStr);
            PhoneParseResult parseResult = PhoneParser.RUSSIA.parse(session.logger(), phoneStr);
            if (parseResult.status() != PhoneParseStatus.OK) {
                PhoneParser.completePhoneChangeStatusParseFailureResponse(parseResult, phoneStr, callback);
                return;
            }
            session.logger().info("Phone parsed: " + phoneStr);
            PhoneContext context =
                new PhoneContext(proxy, session, parseResult.phone(), phoneStr, skipActualEdna);
            StorageRequestUtils.searchRequest(context, proxy, new SearchCallback(context, callback));
        } catch (Exception e) {
            callback.failed(e);
        }
    }

    private final class SearchCallback
        implements FutureCallback<JsonObject> {
        private final FutureCallback<PhoneChangeStatusResponse> callback;
        private final PhoneContext context;

        private SearchCallback(
            final PhoneContext context,
            final AbstractProxySessionCallback<PhoneChangeStatusResponse> callback) {
            this.context = context;
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject searchResult) {
            context.logger().info("Finished sequentual");
            try {
                PhoneInfo phoneInfo = StorageRequestUtils.parseSearchResult(searchResult, context);

                if (phoneInfo == null || (!context.skipEdna() && !phoneInfo.onMonitoring())) {
                    callback.completed(
                        new PhoneChangeStatusResponse(
                            PhoneCheckStatus.NOT_MONITORED,
                            0));
                    return;
                }

                if (context.skipEdna() || context.allowCached()) {
                    callback.completed(
                        new PhoneChangeStatusResponse(
                            PhoneCheckStatus.OK,
                            phoneInfo.lastChangeTime()));
                    return;
                }

                ProcessImsiResponseCallback ednaCallback =
                    new ProcessImsiResponseCallback(context, phoneInfo, callback);

                EdnaRequestUtils.sendImsiRequest(proxy, context, ednaCallback);
            } catch (Exception e) {
                failed(e);
            }
        }

        @Override
        public void failed(Exception e) {
            context.logger().info("Failed sequentual");
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            context.logger().info("Cancelled sequentual");
            callback.cancelled();
        }
    }
}
