package ru.yandex.passport.phone.ownership;

import java.io.IOException;

import ru.yandex.collection.Pattern;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.passport.phone.ownership.config.ImmutablePhoneOwnershipProxyConfig;
import ru.yandex.passport.phone.ownership.edna.PhoneManualEdnaUpdateHandler;
import ru.yandex.passport.phone.ownership.parse.MobileOperatorMapping;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;

public class PhoneOwnershipProxy
    extends UniversalSearchProxy<ImmutablePhoneOwnershipProxyConfig> {
    public static final String PHONE_QUEUE_NAME = System.getProperty("PO_QUEUE");

    private final AsyncClient ednaClient;
    private final MobileOperatorMapping operatorMapping;
    private final SquatterStat squatterStat;

    public PhoneOwnershipProxy(final ImmutablePhoneOwnershipProxyConfig config) throws IOException {
        super(config);

        operatorMapping = new MobileOperatorMapping(this.logger());
        ednaClient = client("EdnaClient", config.ednaConfig());

        this.register(
            new Pattern<>("/phone/change/status", false),
            new PhoneChangeStatusHandler(this));
        this.register(
            new Pattern<>("/phone/tracking/start", false),
            new PhoneStartTrackingHandler(this));
        this.register(
            new Pattern<>("/phone/tracking/stop", false),
            new PhoneUnsubsribeHandler(this));

        this.register(
            new Pattern<>("/phone/manual/update", false),
            new UpdatePhoneStatusHandler(this));

        this.register(
            new Pattern<>("/phone/manual/edna/update", false),
            new PhoneManualEdnaUpdateHandler(this));

        this.squatterStat = new SquatterStat(this);
        registerStater(squatterStat);
    }

    public AsyncClient ednaClient() {
        return ednaClient;
    }

    public static Long normalizePhone(final String phone) throws BadRequestException {
        String phoneStr = phone;
        if (phoneStr.startsWith("+")) {
            phoneStr = phoneStr.substring(1);
        }

        try {
            return Long.parseLong(phoneStr);
        } catch (NumberFormatException nfe) {
            throw new BadRequestException("Wrong phone format");
        }
    }

    public static String phoneId(final Long phone) {
        return "phone_" + phone;
    }

    public MobileOperatorMapping operatorMapping() {
        return operatorMapping;
    }

    @Override
    public void start() throws IOException {
        super.start();

        squatterStat.start();
    }
}
