package ru.yandex.passport.phone.ownership;

import java.util.HashMap;
import java.util.Map;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.passport.phone.ownership.parse.PhoneParseResult;
import ru.yandex.passport.phone.ownership.parse.PhoneParseStatus;
import ru.yandex.passport.phone.ownership.parse.PhoneParser;

public class PhoneStartTrackingDraftFalseHandler implements ProxyRequestHandler {
    private final PhoneOwnershipProxy proxy;
    private final boolean skipActualEdna;

    public PhoneStartTrackingDraftFalseHandler(final PhoneOwnershipProxy proxy) {
        this.proxy = proxy;
        this.skipActualEdna =
            Boolean.parseBoolean(
                System.getProperty("SKIP_EDNA", Boolean.TRUE.toString()));
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        HttpRequest request = session.request();
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        HttpEntityEnclosingRequest postRequest = (HttpEntityEnclosingRequest) request;
        HttpEntity entity = postRequest.getEntity();
        PhoneStatusResponsePrinter callback = new PhoneStatusResponsePrinter(session);
        try {
            JsonMap map = TypesafeValueContentHandler.parse(
                CharsetUtils.content(entity)).asMap();
            String phoneStr = map.getString("phone");
            session.connection().setSessionInfo(
                "phoneStr",
                phoneStr);
            PhoneParseResult parseResult = PhoneParser.RUSSIA.parse(session.logger(), phoneStr);

            //https://st.yandex-team.ru/PASSP-37610
            if (parseResult.phone().getCountryCode() == 7 && phoneStr.startsWith("+7000")) {
                callback.completed(
                    new PhoneStatusResponse(
                        PhoneCheckStatus.OK));
                return;
            }

            if (parseResult.status() != PhoneParseStatus.OK) {
                callback.completed(
                    new PhoneStatusResponse(
                        PhoneCheckStatus.UNTRACKABLE,
                        "Failed to parse phone " + parseResult.reason()));
                return;
            }
            session.logger().info("Phone parsed: " + phoneStr);
            PhoneContext context =
                new PhoneContext(proxy, session, parseResult.phone(), phoneStr, skipActualEdna);
            StorageRequestUtils.searchRequest(context, proxy, new SearchCallback(context, callback));
        } catch (Exception e) {
            callback.failed(e);
        }
    }

    private final class SearchCallback
        implements FutureCallback<JsonObject> {
        final FutureCallback<PhoneStatusResponse> callback;
        private final PhoneContext context;

        private SearchCallback(
            final PhoneContext context,
            final AbstractProxySessionCallback<PhoneStatusResponse> callback) {
            this.context = context;
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject searchResult) {
            context.logger().info("Finished sequentual");
            try {
                PhoneInfo phoneInfo = StorageRequestUtils.parseSearchResult(searchResult, context);
                if (phoneInfo == null) {
                    if (skipActualEdna) {
                        StorageRequestUtils.addPhoneToStorage(context,
                            new PhoneInfo(
                                context,
                                false,
                                null,
                                0,
                                ChangeSource.ID_CREATE,
                                false), proxy, EmptyFutureCallback.INSTANCE);
                        callback.completed(new PhoneStatusResponse(PhoneCheckStatus.OK));
                    } else {
                        context.logger().warning("Bad case, no number in storage, registration is not complete");
                        callback.completed(new PhoneStatusResponse(PhoneCheckStatus.NOT_MONITORED));
                    }
                } else {
                    Map<String, Object> fieldsToUpdate = new HashMap<>();
                    fieldsToUpdate.put("draft", false);
                    StorageRequestUtils.updatePhoneFieldsInStorage(context, phoneInfo.phoneNum(), fieldsToUpdate,
                        proxy, EmptyFutureCallback.INSTANCE);
                    callback.completed(new PhoneStatusResponse(PhoneCheckStatus.OK));
                }
            } catch (Exception e) {
                failed(e);
            }
        }

        @Override
        public void failed(Exception e) {
            context.logger().info("Failed sequentual");
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            context.logger().info("Cancelled sequentual");
            callback.cancelled();
        }
    }
}
