package ru.yandex.passport.phone.ownership;

import java.io.IOException;
import java.util.logging.Level;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.StringEntity;
import org.w3c.dom.Document;

import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.passport.phone.ownership.edna.AddressRequestStatus;
import ru.yandex.passport.phone.ownership.edna.EdnaRequestUtils;
import ru.yandex.passport.phone.ownership.edna.EdnaUnsubscribeResponseConsumerFactory;
import ru.yandex.passport.phone.ownership.edna.UnsubscribeResponse;
import ru.yandex.passport.phone.ownership.parse.PhoneParseResult;
import ru.yandex.passport.phone.ownership.parse.PhoneParser;

public class PhoneUnsubsribeHandler implements ProxyRequestHandler {
    private final PhoneOwnershipProxy proxy;
    private final boolean skipActualEdna;

    public PhoneUnsubsribeHandler(final PhoneOwnershipProxy proxy) {
        this.proxy = proxy;
        this.skipActualEdna =
            Boolean.parseBoolean(
                System.getProperty("SKIP_EDNA", Boolean.TRUE.toString()));

    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        HttpRequest request = session.request();
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        HttpEntityEnclosingRequest postRequest = (HttpEntityEnclosingRequest) request;
        HttpEntity entity = postRequest.getEntity();
        PhoneStatusResponsePrinter callback = new PhoneStatusResponsePrinter(session);
        try {
            JsonMap map = TypesafeValueContentHandler.parse(
                CharsetUtils.content(entity)).asMap();
            String phoneStr = map.getString("phone");
            session.connection().setSessionInfo(
                "phoneStr",
                phoneStr);
            PhoneParseResult parseResult = PhoneParser.RUSSIA.parse(session.logger(), phoneStr);
            switch (parseResult.status()) {
                case PARSE_FAILED:
                    callback.completed(
                        new PhoneStatusResponse(
                            PhoneCheckStatus.UNTRACKABLE,
                            "Failed to parse phone " + parseResult.reason()));
                    return;
                case NOT_VALID_FOR_REGION:
                    //https://st.yandex-team.ru/PASSP-37610
                    if (parseResult.phone().getCountryCode() == 7 && phoneStr.startsWith("+7000")) {
                        callback.completed(
                            new PhoneStatusResponse(
                                PhoneCheckStatus.OK));
                        return;
                    }
                    callback.completed(
                        new PhoneStatusResponse(
                            PhoneCheckStatus.UNTRACKABLE,
                            "Region not supported"));
                    return;
            }
            session.logger().info("Phone parsed: " + phoneStr);
            PhoneContext context = new PhoneContext(proxy, session, parseResult.phone(), phoneStr, skipActualEdna);
            if (skipActualEdna) {
                session.response(HttpStatus.SC_OK, "{\"status\":\"ok\"}");
                session.logger().log(Level.INFO, "Response body: " + "{\"status\":\"ok\"}");
                return;
            }
            sendUnsubsribeRequest(context, callback);
        } catch (Exception e) {
            throw new BadRequestException("Can't handle request", e);
        }
    }

    private void sendUnsubsribeRequest(final PhoneContext context, final PhoneStatusResponsePrinter callback)
        throws ParserConfigurationException, IOException, TransformerException, ConfigException {
        String login = proxy.config().ednaConfig().login();
        String password = proxy.config().ednaConfig().password();
        Document doc = EdnaRequestUtils.constructSubscriberAddressListRequest(context.phone(),
            login, password, "unsubscribeSubscriberRequest");
        HttpTargetConfigBuilder httpTargetConfigBuilder = new HttpTargetConfigBuilder();
        httpTargetConfigBuilder.connections(2);
        QueryConstructor qc = new QueryConstructor("/edna/imsi");
        StringEntity entity = DocumentParser.documentToStringEntity(doc);
        AsyncClient ednaClient = proxy.ednaClient().adjust(context.session().context());
        ProcessUnsubscribeResponseCallback processUnsubscribeResponseCallback =
            new ProcessUnsubscribeResponseCallback(callback);
        ednaClient.execute(
            proxy.config().ednaConfig().host(),
            new BasicAsyncRequestProducerGenerator(qc.toString(), entity),
            EdnaUnsubscribeResponseConsumerFactory.OK,
            context.session().listener().adjustContextGenerator(
                ednaClient.httpClientContextGenerator()),
            processUnsubscribeResponseCallback);
    }

    private static class ProcessUnsubscribeResponseCallback
        extends AbstractFilterFutureCallback<UnsubscribeResponse, PhoneStatusResponse> {

        protected ProcessUnsubscribeResponseCallback(final FutureCallback<? super PhoneStatusResponse> callback) {
            super(callback);
        }

        private void sendResponse(final UnsubscribeResponse response) throws IOException {
            if (response.code().equals(AddressRequestStatus.OK) &&
                response.subscriberCode().equals(AddressRequestStatus.ERROR_NOT_SUBSRIBED)) {
                failed(new BadRequestException("Phone is not subscribed"));
                return;
            }
            if (!(response.code().equals(AddressRequestStatus.OK) &&
                response.subscriberCode().equals(AddressRequestStatus.OK))) {
                failed(new BadRequestException("Unsubscription failed"));
                return;
            }
            callback.completed(new PhoneStatusResponse(PhoneCheckStatus.OK));
        }

        @Override
        public void completed(final UnsubscribeResponse response) {
            try {
                sendResponse(response);
            } catch (IOException e) {
                failed(e);
            }
        }

        @Override
        public void failed(Exception e) {
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }
    }
}
