package ru.yandex.passport.phone.ownership;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;

import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;

import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.searchmap.SearchMapHost;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class SquatterStat extends TimerTask implements Stater {
    public static final long INTERVAL = TimeUnit.MINUTES.toMillis(5);
    private final CloseableHttpClient  client;
    private final PrefixedLogger logger;
    private final Timer timer;
    private final AtomicReference<Map<String, Long>> stat
        = new AtomicReference<>(Collections.emptyMap());
    private final PhoneOwnershipProxy proxy;

    public SquatterStat(
        final PhoneOwnershipProxy proxy)
        throws IOException
    {
        this.logger = proxy.logger().addPrefix("FieldGroupingStater");
        try {
            client = ClientBuilder.createClient(
                new HttpTargetConfigBuilder().connections(2).timeout(60000).build(),
                proxy.config().dnsConfig());
        } catch (ConfigException ce) {
            throw new IOException(ce);
        }
        this.timer = new Timer("FieldGroupingStaterTimer", true);
        this.proxy = proxy;
    }

    public void start() {
        this.timer.scheduleAtFixedRate(this, 1000, INTERVAL);
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        Map<String, Long> data = stat.get();
        for (Map.Entry<String, Long> item: data.entrySet()) {
            statsConsumer.stat(item.getKey(), item.getValue());
        }
    }

    private boolean gatherOnMonitoring(final SearchMapHost host, final Map<String, Long> result) {
        try (CloseableHttpResponse response =
                 client.execute(host.searchHost(), new HttpGet("/search?text=imsi:*&length=0&get=id")))
        {
            if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                JsonObject jo = TypesafeValueContentHandler.parse(CharsetUtils.content(response.getEntity()));
                result.put("on_monitoring_ammx", jo.asMap().getLong("hitsCount"));
                return true;
            } else {
                logger.warning(
                    "Stat Request failed " + response.getStatusLine().getStatusCode() + " " + CharsetUtils.toString(response.getEntity()));
            }
        } catch (Exception e) {
            logger.log(Level.WARNING, "Stat failed", e);
        }

        result.put("on_monitoring_ammx", -1L);
        return false;
    }

    private boolean gatherError(final SearchMapHost host, final Map<String, Long> result) {
        try (CloseableHttpResponse response =
                 client.execute(host.searchHost(), new HttpGet("/search?text=last_edna_request_error_type:*&length=0&get=id")))
        {
            if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                JsonObject jo = TypesafeValueContentHandler.parse(CharsetUtils.content(response.getEntity()));
                result.put("edna_error_ammx", jo.asMap().getLong("hitsCount"));
                return true;
            } else {
                logger.warning(
                    "Stat Request failed " + response.getStatusLine().getStatusCode() + " " + CharsetUtils.toString(response.getEntity()));
            }
        } catch (Exception e) {
            logger.log(Level.WARNING, "Stat failed", e);
        }

        result.put("edna_error_ammx", -1L);
        return false;
    }

    @Override
    public void run() {
        logger.info("Updating  stat");
        List<SearchMapHost> smHost = new ArrayList<>(proxy.searchMap().hosts());
        Collections.shuffle(smHost);
        SearchMapHost host = smHost.iterator().next();
        Map<String, Long> result = new LinkedHashMap<>();

        gatherOnMonitoring(host, result);
        gatherError(host, result);
        stat.set(result);

        logger.info("Updating  finished");
    }
}
